<?php

use Livewire\Component;

new class extends Component {
    public array $section = [];
    public array $items = [];
    public array $company = [];
    public array $social = [];
    public array $services = [];
    public bool $isWhiteBackground = false;
    public ?int $paddingTop = null;
    public ?int $paddingBottom = null;

    public function mount(
        array $section = [],
        array $items = [],
        array $company = [],
        array $social = [],
        array $services = [],
        bool $isWhiteBackground = false,
        ?int $paddingTop = null,
        ?int $paddingBottom = null
    ) {
        $this->company = $company;
        $this->social = $social;
        $this->services = $services;
        $this->isWhiteBackground = $isWhiteBackground;
        $this->paddingTop = $paddingTop;
        $this->paddingBottom = $paddingBottom;

        // Default items with nested features array
        $defaultItems = [
            [
                'name' => 'Basic',
                'price' => '29',
                'period' => 'per month',
                'description' => 'Perfect for individuals',
                'features' => [
                    ['icon' => 'lucide:scale', 'text' => '15 lbs of laundry', 'is_included' => true],
                    ['icon' => 'lucide:sparkles', 'text' => 'Wash & fold service', 'is_included' => true],
                    ['icon' => 'lucide:truck', 'text' => 'Free pickup & delivery', 'is_included' => true],
                    ['icon' => 'lucide:clock', 'text' => '48-hour turnaround', 'is_included' => true],
                    ['icon' => 'lucide:flask-conical', 'text' => 'Standard detergents', 'is_included' => true],
                    ['icon' => 'lucide:leaf', 'text' => 'Premium eco detergents', 'is_included' => false],
                    ['icon' => 'lucide:zap', 'text' => 'Same-day turnaround', 'is_included' => false],
                ],
                'popular' => false,
            ],
            [
                'name' => 'Family',
                'price' => '79',
                'period' => 'per month',
                'description' => 'Best for families',
                'features' => [
                    ['icon' => 'lucide:scale', 'text' => '50 lbs of laundry', 'is_included' => true],
                    ['icon' => 'lucide:sparkles', 'text' => 'Wash, fold & iron', 'is_included' => true],
                    ['icon' => 'lucide:truck', 'text' => 'Free pickup & delivery', 'is_included' => true],
                    ['icon' => 'lucide:clock', 'text' => '24-hour turnaround', 'is_included' => true],
                    ['icon' => 'lucide:leaf', 'text' => 'Premium eco detergents', 'is_included' => true],
                    ['icon' => 'lucide:scissors', 'text' => '2 dry clean items/month', 'is_included' => true],
                    ['icon' => 'lucide:zap', 'text' => 'Same-day turnaround', 'is_included' => false],
                ],
                'popular' => true,
            ],
            [
                'name' => 'Business',
                'price' => '199',
                'period' => 'per month',
                'description' => 'For businesses & hotels',
                'features' => [
                    ['icon' => 'lucide:infinity', 'text' => 'Unlimited laundry', 'is_included' => true],
                    ['icon' => 'lucide:badge-check', 'text' => 'All services included', 'is_included' => true],
                    ['icon' => 'lucide:calendar-days', 'text' => 'Daily pickup available', 'is_included' => true],
                    ['icon' => 'lucide:zap', 'text' => 'Same-day turnaround', 'is_included' => true],
                    ['icon' => 'lucide:user-circle', 'text' => 'Dedicated account manager', 'is_included' => true],
                    ['icon' => 'lucide:phone', 'text' => 'Priority support', 'is_included' => true],
                    ['icon' => 'lucide:paintbrush', 'text' => 'Custom branding', 'is_included' => true],
                ],
                'popular' => false,
            ],
        ];

        // Process items to ensure features is always a nested array
        if (!empty($items)) {
            $this->items = array_map(function ($item) {
                if (isset($item['features'])) {
                    if (is_string($item['features'])) {
                        // Handle legacy newline-separated string (from old textarea input)
                        $lines = array_filter(
                            array_map('trim', explode("\n", $item['features'])),
                            fn($f) => !empty($f)
                        );
                        $item['features'] = array_map(fn($line) => ['icon' => '', 'text' => $line, 'is_included' => true], $lines);
                    } elseif (is_array($item['features'])) {
                        // Check if it's already nested array format or simple array
                        $item['features'] = array_map(function ($feature) {
                            if (is_string($feature)) {
                                // Simple string array - convert to nested format
                                return ['icon' => '', 'text' => $feature, 'is_included' => true];
                            }
                            // Already nested format - normalize field names
                            return [
                                'icon' => $feature['icon'] ?? '',
                                'text' => $feature['text'] ?? '',
                                'is_included' => $feature['is_included'] ?? $feature['included'] ?? true,
                            ];
                        }, $item['features']);
                    }
                } else {
                    $item['features'] = [];
                }
                return $item;
            }, $items);
        } else {
            $this->items = $defaultItems;
        }

        $this->section = array_merge([
            'badge' => 'Pricing Plans',
            'title' => 'Choose Your Perfect Plan',
            'subtitle' => 'Flexible plans designed to fit your lifestyle.',
            'currency_symbol' => '$',
            'popular_badge' => 'Most Popular',
            'cta_button' => 'Get Started',
            'cta_button_url' => '#contact',
        ], $section);
    }

    public function getCustomPaddingStyle(): string
    {
        $styles = [];
        if ($this->paddingTop !== null) {
            $styles[] = "padding-top: {$this->paddingTop}px";
        }
        if ($this->paddingBottom !== null) {
            $styles[] = "padding-bottom: {$this->paddingBottom}px";
        }
        return !empty($styles) ? implode('; ', $styles) : '';
    }
};
?>

<div>
<section id="pricing" class="section {{ $isWhiteBackground ? 'bg-white dark:bg-gray-800' : 'bg-gray-50 dark:bg-gray-900' }}" @if($this->getCustomPaddingStyle()) style="{{ $this->getCustomPaddingStyle() }}" @endif>
    <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        {{-- Section Header --}}
        <div class="text-center mb-16">
            <span class="badge badge-primary mb-4">{{ $section['badge'] }}</span>
            <h2 class="section-title">{{ $section['title'] }}</h2>
            <p class="section-subtitle">{{ $section['subtitle'] }}</p>
        </div>

        {{-- Pricing Cards --}}
        <div class="grid md:grid-cols-3 gap-8 items-stretch">
            @foreach ($items as $index => $plan)
                <div class="scroll-animate pricing-card {{ $plan['popular'] ? 'popular shadow-2xl' : 'shadow-lg' }}">
                    @if ($plan['popular'])
                        <div class="absolute -top-4 left-1/2 -translate-x-1/2">
                            <span class="badge badge-popular px-4 py-1">{{ $section['popular_badge'] }}</span>
                        </div>
                    @endif

                    <div class="text-center mb-8">
                        <h3 class="pricing-plan-name {{ $plan['popular'] ? 'popular' : '' }}">
                            {{ $plan['name'] }}
                        </h3>
                        <p class="pricing-plan-desc {{ $plan['popular'] ? 'popular' : '' }}">
                            {{ $plan['description'] }}
                        </p>
                        <div class="flex items-baseline justify-center">
                            <span class="pricing-plan-price {{ $plan['popular'] ? 'popular' : '' }}">
                                {{ $section['currency_symbol'] ?? '$' }}{{ $plan['price'] }}
                            </span>
                            <span class="pricing-plan-period {{ $plan['popular'] ? 'popular' : '' }}">
                                /{{ $plan['period'] }}
                            </span>
                        </div>
                    </div>

                    <ul class="pricing-features-list">
                        @foreach ($plan['features'] as $feature)
                            @php
                                $featureIcon = is_array($feature) ? ($feature['icon'] ?? '') : '';
                                $featureText = is_array($feature) ? ($feature['text'] ?? '') : $feature;
                                $isIncluded = is_array($feature) ? ($feature['is_included'] ?? $feature['included'] ?? true) : true;
                            @endphp
                            <li class="pricing-feature-item {{ !$isIncluded ? 'excluded' : '' }} {{ $plan['popular'] ? 'popular' : '' }}">
                                <span class="pricing-feature-icon {{ !$isIncluded ? 'excluded' : '' }} {{ $plan['popular'] ? 'popular' : '' }}">
                                    @if (!empty($featureIcon))
                                        <x-icon icon="{{ $featureIcon }}" />
                                    @else
                                        @if ($isIncluded)
                                            <x-icon icon="lucide:check" />
                                        @else
                                            <x-icon icon="lucide:x" />
                                        @endif
                                    @endif
                                </span>
                                <span class="pricing-feature-text {{ !$isIncluded ? 'excluded' : '' }} {{ $plan['popular'] ? 'popular' : '' }}">
                                    {{ $featureText }}
                                </span>
                            </li>
                        @endforeach
                    </ul>

                    <a href="{{ $section['cta_button_url'] ?? '#contact' }}" class="btn w-full {{ $plan['popular'] ? 'btn-white' : 'btn-primary' }}">
                        {{ $section['cta_button'] }}
                    </a>
                </div>
            @endforeach
        </div>
    </div>
</section>

<style>
    /* ===== Pricing Plan Header ===== */
    .pricing-plan-name {
            font-size: 1.25rem;
            font-weight: 700;
            margin-bottom: 0.5rem;
            color: var(--color-gray-900);
        }

        .pricing-plan-name.popular {
            color: #ffffff;
        }

        .dark .pricing-plan-name {
            color: #ffffff;
        }

        .pricing-plan-desc {
            font-size: 0.875rem;
            margin-bottom: 1rem;
            color: var(--color-gray-600);
        }

        .pricing-plan-desc.popular {
            color: rgba(255, 255, 255, 0.8);
        }

        .dark .pricing-plan-desc {
            color: var(--color-gray-400);
        }

        .pricing-plan-price {
            font-size: 2.5rem;
            font-weight: 700;
            color: var(--color-gray-900);
        }

        .pricing-plan-price.popular {
            color: #ffffff;
        }

        .dark .pricing-plan-price {
            color: #ffffff;
        }

        .pricing-plan-period {
            margin-left: 0.5rem;
            color: var(--color-gray-600);
        }

        .pricing-plan-period.popular {
            color: rgba(255, 255, 255, 0.8);
        }

        .dark .pricing-plan-period {
            color: var(--color-gray-400);
        }

        /* ===== Pricing Features List ===== */
        .pricing-features-list {
            display: flex;
            flex-direction: column;
            gap: 0.875rem;
            margin-bottom: 2rem;
            flex: 1;
        }

        /* ===== Pricing Feature Item ===== */
        .pricing-feature-item {
            display: flex;
            align-items: center;
            gap: 0.75rem;
        }

        .pricing-feature-item.excluded {
            opacity: 0.6;
        }

        /* ===== Pricing Feature Icon ===== */
        .pricing-feature-icon {
            display: flex;
            align-items: center;
            justify-content: center;
            width: 1.5rem;
            height: 1.5rem;
            flex-shrink: 0;
        }

        .pricing-feature-icon svg {
            width: 1.25rem;
            height: 1.25rem;
        }

        /* Light mode - Regular cards */
        .pricing-feature-icon {
            color: var(--color-primary-500);
        }

        .pricing-feature-icon.excluded {
            color: var(--color-gray-400);
        }

        /* Light mode - Popular card */
        .pricing-feature-icon.popular {
            color: var(--color-secondary-300, #fcd34d);
        }

        .pricing-feature-icon.popular.excluded {
            color: rgba(255, 255, 255, 0.4);
        }

        /* Dark mode - Regular cards */
        .dark .pricing-feature-icon {
            color: var(--color-primary-400);
        }

        .dark .pricing-feature-icon.excluded {
            color: var(--color-gray-500);
        }

        /* Dark mode - Popular card (same as light mode popular) */
        .dark .pricing-feature-icon.popular {
            color: var(--color-secondary-300, #fcd34d);
        }

        .dark .pricing-feature-icon.popular.excluded {
            color: rgba(255, 255, 255, 0.4);
        }

        /* ===== Pricing Feature Text ===== */
        .pricing-feature-text {
            font-size: 0.9375rem;
            line-height: 1.4;
            color: var(--color-gray-700);
        }

        .pricing-feature-text.excluded {
            text-decoration: line-through;
            color: var(--color-gray-400);
        }

        /* Light mode - Popular card */
        .pricing-feature-text.popular {
            color: rgba(255, 255, 255, 0.9);
        }

        .pricing-feature-text.popular.excluded {
            color: rgba(255, 255, 255, 0.5);
        }

        /* Dark mode - Regular cards */
        .dark .pricing-feature-text {
            color: var(--color-gray-300);
        }

        .dark .pricing-feature-text.excluded {
            color: var(--color-gray-500);
        }

        /* Dark mode - Popular card (same as light mode popular) */
        .dark .pricing-feature-text.popular {
            color: rgba(255, 255, 255, 0.9);
        }

        .dark .pricing-feature-text.popular.excluded {
            color: rgba(255, 255, 255, 0.5);
        }

        /* ===== RTL Support ===== */
        [dir="rtl"] .pricing-feature-item {
            flex-direction: row-reverse;
        }

        [dir="rtl"] .pricing-plan-period {
            margin-left: 0;
            margin-right: 0.5rem;
    }
</style>
</div>
