<?php

use Livewire\Component;

new class extends Component {
    public array $section = [];
    public array $items = [];
    public array $company = [];
    public array $social = [];
    public array $services = [];
    public bool $isWhiteBackground = false;
    public ?int $paddingTop = null;
    public ?int $paddingBottom = null;

    // Default static data for services
    private array $defaultItems = [
        [
            'icon' => 'lucide:shirt',
            'title' => 'Wash & Fold',
            'description' => 'Professional washing, drying, and folding service for your everyday clothes.',
            'price' => 'From $2.50/lb',
            'features' => [
                ['text' => 'Same-day service available'],
                ['text' => 'Eco-friendly detergents'],
                ['text' => 'Fabric softener included'],
            ],
            'button_text' => 'Learn More',
            'button_url' => '#contact',
            'button_icon' => 'lucide:arrow-right',
        ],
        [
            'icon' => 'lucide:sparkles',
            'title' => 'Dry Cleaning',
            'description' => 'Expert care for your delicate and special garments that need extra attention.',
            'price' => 'From $8.99/item',
            'features' => [
                ['text' => 'Suits & formal wear'],
                ['text' => 'Delicate fabrics'],
                ['text' => 'Stain treatment included'],
            ],
            'button_text' => 'Learn More',
            'button_url' => '#contact',
            'button_icon' => 'lucide:arrow-right',
        ],
        [
            'icon' => 'lucide:wind',
            'title' => 'Press & Iron',
            'description' => 'Professional pressing and ironing for a crisp, polished look.',
            'price' => 'From $3.50/item',
            'features' => [
                ['text' => 'Steam pressing'],
                ['text' => 'Hand finishing'],
                ['text' => 'Ready to wear'],
            ],
            'button_text' => 'Learn More',
            'button_url' => '#contact',
            'button_icon' => 'lucide:arrow-right',
        ],
        [
            'icon' => 'lucide:bed-double',
            'title' => 'Bedding & Linens',
            'description' => 'Fresh, clean bedding and household linens for maximum comfort.',
            'price' => 'From $15.99/set',
            'features' => [
                ['text' => 'Comforters & duvets'],
                ['text' => 'Sheets & pillowcases'],
                ['text' => 'Allergen removal'],
            ],
            'button_text' => 'Learn More',
            'button_url' => '#contact',
            'button_icon' => 'lucide:arrow-right',
        ],
        [
            'icon' => 'lucide:gem',
            'title' => 'Premium Care',
            'description' => 'Specialized treatment for luxury fabrics and designer pieces.',
            'price' => 'Custom pricing',
            'features' => [
                ['text' => 'Designer labels'],
                ['text' => 'Silk & cashmere'],
                ['text' => 'Hand wash only items'],
            ],
            'button_text' => 'Learn More',
            'button_url' => '#contact',
            'button_icon' => 'lucide:arrow-right',
        ],
        [
            'icon' => 'lucide:zap',
            'title' => 'Express Service',
            'description' => 'Urgent cleaning with quick turnaround when you need it fast.',
            'price' => '+50% rush fee',
            'features' => [
                ['text' => '2-hour turnaround'],
                ['text' => 'Priority handling'],
                ['text' => 'Delivery included'],
            ],
            'button_text' => 'Learn More',
            'button_url' => '#contact',
            'button_icon' => 'lucide:arrow-right',
        ],
    ];

    public function mount(
        array $section = [],
        array $items = [],
        array $company = [],
        array $social = [],
        array $services = [],
        bool $isWhiteBackground = false,
        ?int $paddingTop = null,
        ?int $paddingBottom = null
    ) {
        $this->company = $company;
        $this->social = $social;
        $this->isWhiteBackground = $isWhiteBackground;
        $this->paddingTop = $paddingTop;
        $this->paddingBottom = $paddingBottom;

        // Build section data with defaults
        $this->section = array_merge([
            'badge' => 'Our Services',
            'title' => 'Professional Laundry Services',
            'subtitle' => 'From everyday laundry to delicate dry cleaning, we\'ve got all your garment care needs covered.',
        ], $section);

        // Use items from section content (CMS), then $items prop, then default static data
        $this->items = !empty($section['items']) ? $section['items'] : (!empty($items) ? $items : $this->defaultItems);
    }

    /**
     * Get custom padding style if set
     */
    public function getCustomPaddingStyle(): string
    {
        $styles = [];
        if ($this->paddingTop !== null) {
            $styles[] = "padding-top: {$this->paddingTop}px";
        }
        if ($this->paddingBottom !== null) {
            $styles[] = "padding-bottom: {$this->paddingBottom}px";
        }
        return !empty($styles) ? implode('; ', $styles) : '';
    }
};
?>

<div>
<section id="services" class="services-cards-section {{ $isWhiteBackground ? 'services-cards-white' : 'services-cards-gray' }}" @if($this->getCustomPaddingStyle()) style="{{ $this->getCustomPaddingStyle() }}" @endif>
    {{-- Background Pattern --}}
    <div class="services-cards-bg">
        <div class="services-cards-pattern"></div>
        <div class="services-cards-shape services-cards-shape-1"></div>
        <div class="services-cards-shape services-cards-shape-2"></div>
    </div>

    <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 relative">
        {{-- Section Header --}}
        <div class="services-cards-header">
            <div class="services-cards-badge">
                <x-icon icon="lucide:sparkles" class="w-4 h-4" />
                <span>{{ $section['badge'] }}</span>
            </div>
            <h2 class="services-cards-title">{{ $section['title'] }}</h2>
            <p class="services-cards-subtitle">{{ $section['subtitle'] }}</p>
        </div>

        {{-- Services Grid --}}
        <div class="services-cards-grid">
            @foreach ($items as $index => $item)
                <div class="services-card-item scroll-animate" style="animation-delay: {{ $index * 100 }}ms">
                    {{-- Card Top Accent --}}
                    <div class="services-card-accent"></div>

                    {{-- Card Header --}}
                    <div class="services-card-header">
                        <div class="services-card-icon">
                            <x-icon icon="{{ $item['icon'] ?? 'heroicons:sparkles' }}" class="w-7 h-7" />
                        </div>
                        @if (!empty($item['price']))
                            <div class="services-card-price">{{ $item['price'] }}</div>
                        @endif
                    </div>

                    {{-- Card Body --}}
                    <div class="services-card-body">
                        <h3 class="services-card-title">{{ $item['title'] ?? '' }}</h3>
                        <p class="services-card-desc">{{ $item['description'] ?? '' }}</p>

                        {{-- Features --}}
                        @if (!empty($item['features']))
                            <div class="services-card-features">
                                @foreach ($item['features'] as $feature)
                                    <div class="services-card-feature">
                                        <x-icon icon="lucide:check-circle" class="w-4 h-4" />
                                        <span>{{ is_array($feature) ? ($feature['text'] ?? '') : $feature }}</span>
                                    </div>
                                @endforeach
                            </div>
                        @endif
                    </div>

                    {{-- Card Footer --}}
                    @if (!empty($item['button_text']) && !empty($item['button_url']))
                        <div class="services-card-footer">
                            <a href="{{ $item['button_url'] }}" class="services-card-button">
                                <span>{{ $item['button_text'] }}</span>
                                <x-icon icon="{{ $item['button_icon'] ?? 'lucide:arrow-right' }}" class="w-4 h-4" />
                            </a>
                        </div>
                    @endif
                </div>
            @endforeach
        </div>
    </div>
</section>

<style>
    /* ===== Services Cards Section ===== */
    .services-cards-section {
        position: relative;
        padding: 6rem 0;
        overflow: hidden;
    }

    .services-cards-white {
        background: #ffffff;
    }

    .services-cards-gray {
        background: #f8fafc;
    }

    .dark .services-cards-white {
        background: var(--color-bg-secondary);
    }

    .dark .services-cards-gray {
        background: var(--color-bg-primary);
    }

    /* ===== Background ===== */
    .services-cards-bg {
        position: absolute;
        inset: 0;
        pointer-events: none;
        overflow: hidden;
    }

    .services-cards-pattern {
        position: absolute;
        inset: 0;
        background-image: radial-gradient(circle at 1px 1px, var(--color-gray-200) 1px, transparent 0);
        background-size: 40px 40px;
        opacity: 0.5;
    }

    .dark .services-cards-pattern {
        background-image: radial-gradient(circle at 1px 1px, var(--color-gray-700) 1px, transparent 0);
        opacity: 0.3;
    }

    .services-cards-shape {
        position: absolute;
        border-radius: 50%;
        filter: blur(100px);
    }

    .services-cards-shape-1 {
        width: 400px;
        height: 400px;
        background: var(--color-primary-200);
        opacity: 0.3;
        top: -100px;
        right: -100px;
    }

    .services-cards-shape-2 {
        width: 300px;
        height: 300px;
        background: var(--color-secondary-200);
        opacity: 0.3;
        bottom: -50px;
        left: -50px;
    }

    .dark .services-cards-shape-1 {
        background: var(--color-primary-800);
        opacity: 0.2;
    }

    .dark .services-cards-shape-2 {
        background: var(--color-secondary-800);
        opacity: 0.2;
    }

    /* ===== Header ===== */
    .services-cards-header {
        text-align: center;
        margin-bottom: 4rem;
    }

    .services-cards-badge {
        display: inline-flex;
        align-items: center;
        gap: 0.5rem;
        padding: 0.5rem 1rem;
        background: white;
        border: 1px solid var(--color-gray-200);
        border-radius: 9999px;
        font-size: 0.875rem;
        font-weight: 600;
        color: var(--color-primary-600);
        margin-bottom: 1.5rem;
        box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.05);
    }

    .dark .services-cards-badge {
        background: var(--color-bg-tertiary);
        border-color: var(--color-border);
        color: var(--color-primary-400);
    }

    .services-cards-title {
        font-family: var(--font-heading);
        font-size: 2.5rem;
        font-weight: 800;
        color: var(--color-gray-900);
        margin-bottom: 1rem;
    }

    .dark .services-cards-title {
        color: var(--color-text-primary);
    }

    .services-cards-subtitle {
        font-size: 1.125rem;
        color: var(--color-gray-600);
        max-width: 600px;
        margin: 0 auto;
    }

    .dark .services-cards-subtitle {
        color: var(--color-text-tertiary);
    }

    /* ===== Grid ===== */
    .services-cards-grid {
        display: grid;
        gap: 2rem;
    }

    @media (min-width: 640px) {
        .services-cards-grid {
            grid-template-columns: repeat(2, 1fr);
        }
    }

    @media (min-width: 1024px) {
        .services-cards-grid {
            grid-template-columns: repeat(3, 1fr);
        }
    }

    /* ===== Card Item ===== */
    .services-card-item {
        position: relative;
        background: white;
        border-radius: 1.25rem;
        box-shadow: 0 10px 40px -10px rgba(0, 0, 0, 0.1);
        overflow: hidden;
        display: flex;
        flex-direction: column;
        transition: all 0.4s cubic-bezier(0.4, 0, 0.2, 1);
    }

    .services-card-item:hover {
        transform: translateY(-8px);
        box-shadow: 0 25px 50px -12px rgba(0, 0, 0, 0.15);
    }

    .dark .services-card-item {
        background: var(--color-card-bg);
        box-shadow: 0 10px 40px -10px rgba(0, 0, 0, 0.3);
    }

    .dark .services-card-item:hover {
        box-shadow: 0 25px 50px -12px rgba(0, 0, 0, 0.5);
    }

    /* ===== Card Accent ===== */
    .services-card-accent {
        height: 4px;
        background: linear-gradient(90deg, var(--color-primary-500), var(--color-secondary-500));
    }

    /* ===== Card Header ===== */
    .services-card-header {
        display: flex;
        align-items: center;
        justify-content: space-between;
        padding: 1.5rem 1.5rem 0;
    }

    .services-card-icon {
        width: 3.5rem;
        height: 3.5rem;
        display: flex;
        align-items: center;
        justify-content: center;
        background: linear-gradient(135deg, var(--color-primary-50), var(--color-primary-100));
        border-radius: 1rem;
        color: var(--color-primary-600);
        transition: all 0.3s ease;
    }

    .services-card-item:hover .services-card-icon {
        background: linear-gradient(135deg, var(--color-primary-500), var(--color-primary-600));
        color: white;
        transform: rotate(-10deg) scale(1.1);
    }

    .dark .services-card-icon {
        background: linear-gradient(135deg, rgba(var(--color-primary-rgb, 37, 99, 235), 0.2), rgba(var(--color-primary-rgb, 37, 99, 235), 0.1));
        color: var(--color-primary-400);
    }

    .dark .services-card-item:hover .services-card-icon {
        background: linear-gradient(135deg, var(--color-primary-500), var(--color-primary-600));
        color: white;
    }

    .services-card-price {
        font-family: var(--font-heading);
        font-size: 0.875rem;
        font-weight: 700;
        color: var(--color-primary-600);
        padding: 0.375rem 0.875rem;
        background: var(--color-primary-50);
        border-radius: 9999px;
    }

    .dark .services-card-price {
        background: rgba(var(--color-primary-rgb, 37, 99, 235), 0.15);
        color: var(--color-primary-400);
    }

    /* ===== Card Body ===== */
    .services-card-body {
        padding: 1.5rem;
        flex: 1;
    }

    .services-card-title {
        font-family: var(--font-heading);
        font-size: 1.25rem;
        font-weight: 700;
        color: var(--color-gray-900);
        margin-bottom: 0.5rem;
    }

    .dark .services-card-title {
        color: var(--color-text-primary);
    }

    .services-card-desc {
        font-size: 0.9375rem;
        color: var(--color-gray-600);
        line-height: 1.6;
        margin-bottom: 1.25rem;
    }

    .dark .services-card-desc {
        color: var(--color-text-tertiary);
    }

    .services-card-features {
        display: flex;
        flex-direction: column;
        gap: 0.625rem;
    }

    .services-card-feature {
        display: flex;
        align-items: center;
        gap: 0.5rem;
        font-size: 0.875rem;
        color: var(--color-gray-600);
    }

    .services-card-feature svg {
        color: var(--color-primary-500);
        flex-shrink: 0;
    }

    .dark .services-card-feature {
        color: var(--color-text-tertiary);
    }

    .dark .services-card-feature svg {
        color: var(--color-primary-400);
    }

    /* ===== Card Footer ===== */
    .services-card-footer {
        padding: 1rem 1.5rem 1.5rem;
        border-top: 1px solid var(--color-gray-100);
    }

    .dark .services-card-footer {
        border-top-color: var(--color-border);
    }

    .services-card-button {
        display: flex;
        align-items: center;
        justify-content: center;
        gap: 0.5rem;
        width: 100%;
        padding: 0.75rem 1.5rem;
        background: var(--color-gray-900);
        color: white;
        font-size: 0.875rem;
        font-weight: 600;
        border-radius: 0.75rem;
        transition: all 0.3s ease;
    }

    .services-card-button:hover {
        background: var(--color-primary-600);
        transform: translateY(-2px);
    }

    .dark .services-card-button {
        background: var(--color-primary-600);
    }

    .dark .services-card-button:hover {
        background: var(--color-primary-500);
    }

    /* ===== RTL Support ===== */
    [dir="rtl"] .services-cards-section {
        direction: rtl;
    }

    [dir="rtl"] .services-cards-badge {
        flex-direction: row-reverse;
    }

    [dir="rtl"] .services-card-header {
        flex-direction: row-reverse;
    }

    [dir="rtl"] .services-card-body {
        text-align: right;
    }

    [dir="rtl"] .services-card-features {
        align-items: flex-start;
    }

    [dir="rtl"] .services-card-feature {
        flex-direction: row-reverse;
        text-align: right;
    }

    [dir="rtl"] .services-card-button {
        flex-direction: row-reverse;
    }

    [dir="rtl"] .services-card-button svg {
        transform: scaleX(-1);
    }

    [dir="rtl"] .services-card-item:hover .services-card-icon {
        transform: rotate(10deg) scale(1.1);
    }

    [dir="rtl"] .services-cards-shape-1 {
        right: auto;
        left: -100px;
    }

    [dir="rtl"] .services-cards-shape-2 {
        left: auto;
        right: -50px;
    }

    /* ===== Responsive ===== */
    @media (max-width: 640px) {
        .services-cards-section {
            padding: 4rem 0;
        }

        .services-cards-title {
            font-size: 2rem;
        }

        .services-cards-header {
            margin-bottom: 3rem;
        }
    }
</style>
</div>
