<?php

use Livewire\Component;

new class extends Component {
    public array $section = [];
    public array $items = [];
    public array $company = [];
    public array $social = [];
    public array $services = [];
    public bool $isWhiteBackground = false;
    public ?int $paddingTop = null;
    public ?int $paddingBottom = null;

    // Default static data for services
    private array $defaultItems = [
        [
            'icon' => 'lucide:shirt',
            'title' => 'Wash & Fold',
            'description' => 'Professional washing, drying, and folding service for your everyday clothes.',
            'price' => 'From $2.50/lb',
            'features' => [
                ['text' => 'Same-day service available'],
                ['text' => 'Eco-friendly detergents'],
                ['text' => 'Fabric softener included'],
            ],
            'button_text' => 'Learn More',
            'button_url' => '#contact',
            'button_icon' => 'lucide:arrow-right',
        ],
        [
            'icon' => 'lucide:sparkles',
            'title' => 'Dry Cleaning',
            'description' => 'Expert care for your delicate and special garments that need extra attention.',
            'price' => 'From $8.99/item',
            'features' => [
                ['text' => 'Suits & formal wear'],
                ['text' => 'Delicate fabrics'],
                ['text' => 'Stain treatment included'],
            ],
            'button_text' => 'Learn More',
            'button_url' => '#contact',
            'button_icon' => 'lucide:arrow-right',
        ],
        [
            'icon' => 'lucide:wind',
            'title' => 'Press & Iron',
            'description' => 'Professional pressing and ironing for a crisp, polished look.',
            'price' => 'From $3.50/item',
            'features' => [
                ['text' => 'Steam pressing'],
                ['text' => 'Hand finishing'],
                ['text' => 'Ready to wear'],
            ],
            'button_text' => 'Learn More',
            'button_url' => '#contact',
            'button_icon' => 'lucide:arrow-right',
        ],
        [
            'icon' => 'lucide:bed-double',
            'title' => 'Bedding & Linens',
            'description' => 'Fresh, clean bedding and household linens for maximum comfort.',
            'price' => 'From $15.99/set',
            'features' => [
                ['text' => 'Comforters & duvets'],
                ['text' => 'Sheets & pillowcases'],
                ['text' => 'Allergen removal'],
            ],
            'button_text' => 'Learn More',
            'button_url' => '#contact',
            'button_icon' => 'lucide:arrow-right',
        ],
        [
            'icon' => 'lucide:gem',
            'title' => 'Premium Care',
            'description' => 'Specialized treatment for luxury fabrics and designer pieces.',
            'price' => 'Custom pricing',
            'features' => [
                ['text' => 'Designer labels'],
                ['text' => 'Silk & cashmere'],
                ['text' => 'Hand wash only items'],
            ],
            'button_text' => 'Learn More',
            'button_url' => '#contact',
            'button_icon' => 'lucide:arrow-right',
        ],
        [
            'icon' => 'lucide:zap',
            'title' => 'Express Service',
            'description' => 'Urgent cleaning with quick turnaround when you need it fast.',
            'price' => '+50% rush fee',
            'features' => [
                ['text' => '2-hour turnaround'],
                ['text' => 'Priority handling'],
                ['text' => 'Delivery included'],
            ],
            'button_text' => 'Learn More',
            'button_url' => '#contact',
            'button_icon' => 'lucide:arrow-right',
        ],
    ];

    public function mount(
        array $section = [],
        array $items = [],
        array $company = [],
        array $social = [],
        array $services = [],
        bool $isWhiteBackground = false,
        ?int $paddingTop = null,
        ?int $paddingBottom = null
    ) {
        $this->company = $company;
        $this->social = $social;
        $this->isWhiteBackground = $isWhiteBackground;
        $this->paddingTop = $paddingTop;
        $this->paddingBottom = $paddingBottom;

        // Build section data with defaults
        $this->section = array_merge([
            'badge' => 'Our Services',
            'title' => 'Professional Laundry Services',
            'subtitle' => 'From everyday laundry to delicate dry cleaning, we\'ve got all your garment care needs covered.',
        ], $section);

        // Use items from section content (CMS), then $items prop, then default static data
        $this->items = !empty($section['items']) ? $section['items'] : (!empty($items) ? $items : $this->defaultItems);
    }

    /**
     * Get custom padding style if set
     */
    public function getCustomPaddingStyle(): string
    {
        $styles = [];
        if ($this->paddingTop !== null) {
            $styles[] = "padding-top: {$this->paddingTop}px";
        }
        if ($this->paddingBottom !== null) {
            $styles[] = "padding-bottom: {$this->paddingBottom}px";
        }
        return !empty($styles) ? implode('; ', $styles) : '';
    }
};
?>

<div>
<section id="services" class="services-gradient-section" @if($this->getCustomPaddingStyle()) style="{{ $this->getCustomPaddingStyle() }}" @endif>
    {{-- Background --}}
    <div class="services-gradient-bg">
        <div class="services-gradient-mesh"></div>
        <div class="services-gradient-orb services-gradient-orb-1"></div>
        <div class="services-gradient-orb services-gradient-orb-2"></div>
        <div class="services-gradient-orb services-gradient-orb-3"></div>
    </div>

    <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 relative">
        {{-- Section Header --}}
        <div class="services-gradient-header">
            <div class="services-gradient-badge">
                <span class="services-gradient-badge-dot"></span>
                {{ $section['badge'] }}
            </div>
            <h2 class="services-gradient-title">{{ $section['title'] }}</h2>
            <p class="services-gradient-subtitle">{{ $section['subtitle'] }}</p>
        </div>

        {{-- Services Grid --}}
        <div class="services-gradient-grid">
            @foreach ($items as $index => $item)
                <div class="services-gradient-card scroll-animate" style="animation-delay: {{ $index * 100 }}ms">
                    {{-- Gradient Border --}}
                    <div class="services-gradient-card-border"></div>

                    {{-- Card Inner --}}
                    <div class="services-gradient-card-inner">
                        {{-- Top Section with Icon --}}
                        <div class="services-gradient-top">
                            <div class="services-gradient-icon">
                                <x-icon icon="{{ $item['icon'] ?? 'heroicons:sparkles' }}" class="w-7 h-7" />
                            </div>
                            @if (!empty($item['price']))
                                <div class="services-gradient-price">{{ $item['price'] }}</div>
                            @endif
                        </div>

                        {{-- Content --}}
                        <div class="services-gradient-content">
                            <h3 class="services-gradient-card-title">{{ $item['title'] ?? '' }}</h3>
                            <p class="services-gradient-desc">{{ $item['description'] ?? '' }}</p>
                        </div>

                        {{-- Features --}}
                        @if (!empty($item['features']))
                            <div class="services-gradient-features">
                                @foreach ($item['features'] as $feature)
                                    <div class="services-gradient-feature">
                                        <div class="services-gradient-feature-check">
                                            <x-icon icon="lucide:check" class="w-3 h-3" />
                                        </div>
                                        <span>{{ is_array($feature) ? ($feature['text'] ?? '') : $feature }}</span>
                                    </div>
                                @endforeach
                            </div>
                        @endif

                        {{-- Button --}}
                        @if (!empty($item['button_text']) && !empty($item['button_url']))
                            <a href="{{ $item['button_url'] }}" class="services-gradient-btn">
                                <span>{{ $item['button_text'] }}</span>
                                <x-icon icon="{{ $item['button_icon'] ?? 'lucide:arrow-right' }}" class="w-4 h-4" />
                            </a>
                        @endif
                    </div>
                </div>
            @endforeach
        </div>
    </div>
</section>

<style>
    /* ===== Services Gradient Section ===== */
    .services-gradient-section {
        position: relative;
        padding: 6rem 0;
        overflow: hidden;
        background: linear-gradient(135deg, var(--color-primary-50) 0%, var(--color-primary-100) 50%, var(--color-secondary-50) 100%);
    }

    .dark .services-gradient-section {
        background: var(--color-bg-primary);
    }

    /* ===== Background ===== */
    .services-gradient-bg {
        position: absolute;
        inset: 0;
        pointer-events: none;
        overflow: hidden;
    }

    .services-gradient-mesh {
        position: absolute;
        inset: 0;
        background-image: radial-gradient(circle at 2px 2px, var(--color-primary-200) 1px, transparent 0);
        background-size: 50px 50px;
        opacity: 0.5;
    }

    .dark .services-gradient-mesh {
        background-image: radial-gradient(circle at 2px 2px, var(--color-border) 1px, transparent 0);
        opacity: 0.3;
    }

    .services-gradient-orb {
        position: absolute;
        border-radius: 50%;
        filter: blur(100px);
    }

    .services-gradient-orb-1 {
        width: 500px;
        height: 500px;
        background: var(--color-primary-300);
        opacity: 0.4;
        top: -200px;
        left: 10%;
        animation: services-gradient-float 15s ease-in-out infinite;
    }

    .services-gradient-orb-2 {
        width: 400px;
        height: 400px;
        background: var(--color-secondary-300);
        opacity: 0.3;
        bottom: -150px;
        right: 15%;
        animation: services-gradient-float 20s ease-in-out infinite reverse;
    }

    .services-gradient-orb-3 {
        width: 300px;
        height: 300px;
        background: var(--color-primary-200);
        opacity: 0.3;
        top: 50%;
        right: 5%;
        animation: services-gradient-float 12s ease-in-out infinite;
    }

    .dark .services-gradient-orb-1 {
        background: var(--color-primary-800);
        opacity: 0.3;
    }

    .dark .services-gradient-orb-2 {
        background: var(--color-secondary-800);
        opacity: 0.25;
    }

    .dark .services-gradient-orb-3 {
        background: var(--color-primary-700);
        opacity: 0.2;
    }

    @keyframes services-gradient-float {
        0%, 100% { transform: translate(0, 0) rotate(0deg); }
        33% { transform: translate(30px, -30px) rotate(10deg); }
        66% { transform: translate(-20px, 20px) rotate(-5deg); }
    }

    /* ===== Header ===== */
    .services-gradient-header {
        text-align: center;
        margin-bottom: 4rem;
    }

    .services-gradient-badge {
        display: inline-flex;
        align-items: center;
        gap: 0.5rem;
        padding: 0.5rem 1.25rem;
        background: white;
        border: 1px solid var(--color-primary-200);
        backdrop-filter: blur(10px);
        color: var(--color-primary-700);
        font-size: 0.8125rem;
        font-weight: 600;
        border-radius: 9999px;
        margin-bottom: 1.5rem;
        box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.05);
    }

    .dark .services-gradient-badge {
        background: var(--color-bg-tertiary);
        border: 1px solid var(--color-border);
        color: var(--color-primary-400);
        box-shadow: none;
    }

    .services-gradient-badge-dot {
        width: 8px;
        height: 8px;
        background: var(--color-secondary-500);
        border-radius: 50%;
        animation: services-gradient-pulse 2s ease-in-out infinite;
    }

    .dark .services-gradient-badge-dot {
        background: var(--color-primary-500);
    }

    @keyframes services-gradient-pulse {
        0%, 100% { opacity: 1; transform: scale(1); }
        50% { opacity: 0.5; transform: scale(1.2); }
    }

    .services-gradient-title {
        font-family: var(--font-heading);
        font-size: 2.5rem;
        font-weight: 800;
        color: var(--color-gray-900);
        margin-bottom: 1rem;
    }

    .dark .services-gradient-title {
        color: var(--color-text-primary);
    }

    .services-gradient-subtitle {
        font-size: 1.125rem;
        color: var(--color-gray-600);
        max-width: 600px;
        margin: 0 auto;
    }

    .dark .services-gradient-subtitle {
        color: var(--color-text-tertiary);
    }

    /* ===== Grid ===== */
    .services-gradient-grid {
        display: grid;
        gap: 1.5rem;
    }

    @media (min-width: 640px) {
        .services-gradient-grid {
            grid-template-columns: repeat(2, 1fr);
        }
    }

    @media (min-width: 1024px) {
        .services-gradient-grid {
            grid-template-columns: repeat(3, 1fr);
            gap: 2rem;
        }
    }

    /* ===== Card ===== */
    .services-gradient-card {
        position: relative;
        border-radius: 1.25rem;
        padding: 2px;
        transition: all 0.4s cubic-bezier(0.4, 0, 0.2, 1);
    }

    .services-gradient-card:hover {
        transform: translateY(-8px);
    }

    .services-gradient-card-border {
        position: absolute;
        inset: 0;
        border-radius: 1.25rem;
        background: linear-gradient(135deg, var(--color-primary-400), var(--color-secondary-400), var(--color-primary-400));
        background-size: 200% 200%;
        opacity: 0.5;
        transition: opacity 0.3s ease;
    }

    .dark .services-gradient-card-border {
        background: linear-gradient(135deg, var(--color-primary-600), var(--color-secondary-600), var(--color-primary-600));
        opacity: 0.3;
    }

    .services-gradient-card:hover .services-gradient-card-border {
        opacity: 1;
        animation: services-gradient-border 3s linear infinite;
    }

    .dark .services-gradient-card:hover .services-gradient-card-border {
        opacity: 0.6;
    }

    @keyframes services-gradient-border {
        0% { background-position: 0% 50%; }
        50% { background-position: 100% 50%; }
        100% { background-position: 0% 50%; }
    }

    .services-gradient-card-inner {
        position: relative;
        background: white;
        backdrop-filter: blur(20px);
        border-radius: calc(1.25rem - 2px);
        padding: 1.75rem;
        height: 100%;
        display: flex;
        flex-direction: column;
    }

    .dark .services-gradient-card-inner {
        background: var(--color-card-bg);
        border: 1px solid var(--color-border);
    }

    /* ===== Top Section ===== */
    .services-gradient-top {
        display: flex;
        align-items: center;
        justify-content: space-between;
        margin-bottom: 1.25rem;
    }

    .services-gradient-icon {
        width: 3.5rem;
        height: 3.5rem;
        display: flex;
        align-items: center;
        justify-content: center;
        background: linear-gradient(135deg, var(--color-primary-500), var(--color-secondary-500));
        border-radius: 1rem;
        color: white;
        transition: all 0.3s ease;
    }

    .dark .services-gradient-icon {
        background: linear-gradient(135deg, var(--color-primary-600), var(--color-secondary-600));
    }

    .services-gradient-card:hover .services-gradient-icon {
        transform: rotate(-10deg) scale(1.1);
    }

    .services-gradient-price {
        font-family: var(--font-heading);
        font-size: 0.875rem;
        font-weight: 700;
        color: var(--color-primary-700);
        padding: 0.375rem 0.875rem;
        background: var(--color-primary-50);
        border-radius: 9999px;
    }

    .dark .services-gradient-price {
        color: var(--color-primary-400);
        background: rgba(var(--color-primary-rgb, 37, 99, 235), 0.15);
    }

    /* ===== Content ===== */
    .services-gradient-content {
        flex: 1;
        margin-bottom: 1.25rem;
    }

    .services-gradient-card-title {
        font-family: var(--font-heading);
        font-size: 1.25rem;
        font-weight: 700;
        color: var(--color-gray-900);
        margin-bottom: 0.5rem;
    }

    .dark .services-gradient-card-title {
        color: var(--color-text-primary);
    }

    .services-gradient-desc {
        font-size: 0.9375rem;
        color: var(--color-gray-600);
        line-height: 1.6;
    }

    .dark .services-gradient-desc {
        color: var(--color-text-tertiary);
    }

    /* ===== Features ===== */
    .services-gradient-features {
        display: flex;
        flex-direction: column;
        gap: 0.625rem;
        margin-bottom: 1.25rem;
    }

    .services-gradient-feature {
        display: flex;
        align-items: center;
        gap: 0.625rem;
        font-size: 0.875rem;
        color: var(--color-gray-600);
    }

    .dark .services-gradient-feature {
        color: var(--color-text-tertiary);
    }

    .services-gradient-feature-check {
        width: 1.25rem;
        height: 1.25rem;
        display: flex;
        align-items: center;
        justify-content: center;
        background: linear-gradient(135deg, var(--color-primary-500), var(--color-secondary-500));
        border-radius: 50%;
        flex-shrink: 0;
        color: white;
    }

    .dark .services-gradient-feature-check {
        background: linear-gradient(135deg, var(--color-primary-600), var(--color-secondary-600));
    }

    /* ===== Button ===== */
    .services-gradient-btn {
        display: inline-flex;
        align-items: center;
        justify-content: center;
        gap: 0.5rem;
        padding: 0.75rem 1.5rem;
        background: linear-gradient(135deg, var(--color-primary-600), var(--color-secondary-600));
        color: white;
        font-size: 0.875rem;
        font-weight: 600;
        border-radius: 0.75rem;
        transition: all 0.3s ease;
        margin-top: auto;
    }

    .services-gradient-btn:hover {
        background: linear-gradient(135deg, var(--color-primary-700), var(--color-secondary-700));
        transform: translateY(-2px);
        box-shadow: 0 10px 20px -5px rgba(var(--color-primary-rgb, 37, 99, 235), 0.3);
    }

    .dark .services-gradient-btn {
        background: linear-gradient(135deg, var(--color-primary-500), var(--color-secondary-500));
    }

    .dark .services-gradient-btn:hover {
        background: linear-gradient(135deg, var(--color-primary-400), var(--color-secondary-400));
        box-shadow: 0 10px 20px -5px rgba(var(--color-primary-rgb, 37, 99, 235), 0.25);
    }

    /* ===== RTL Support ===== */
    [dir="rtl"] .services-gradient-section {
        direction: rtl;
    }

    [dir="rtl"] .services-gradient-header {
        text-align: center;
    }

    [dir="rtl"] .services-gradient-badge {
        flex-direction: row-reverse;
    }

    [dir="rtl"] .services-gradient-top {
        flex-direction: row-reverse;
    }

    [dir="rtl"] .services-gradient-card-inner {
        text-align: right;
    }

    [dir="rtl"] .services-gradient-content {
        text-align: right;
    }

    [dir="rtl"] .services-gradient-features {
        align-items: flex-start;
    }

    [dir="rtl"] .services-gradient-feature {
        flex-direction: row-reverse;
        text-align: right;
        justify-content: flex-end;
    }

    [dir="rtl"] .services-gradient-btn {
        flex-direction: row-reverse;
    }

    [dir="rtl"] .services-gradient-btn svg {
        transform: scaleX(-1);
    }

    [dir="rtl"] .services-gradient-card:hover .services-gradient-icon {
        transform: rotate(10deg) scale(1.1);
    }

    /* RTL Background Orbs */
    [dir="rtl"] .services-gradient-orb-1 {
        left: auto;
        right: 10%;
    }

    [dir="rtl"] .services-gradient-orb-2 {
        right: auto;
        left: 15%;
    }

    [dir="rtl"] .services-gradient-orb-3 {
        right: auto;
        left: 5%;
    }

    /* ===== Responsive ===== */
    @media (max-width: 640px) {
        .services-gradient-section {
            padding: 4rem 0;
        }

        .services-gradient-title {
            font-size: 2rem;
        }

        .services-gradient-card-inner {
            padding: 1.5rem;
        }

        .services-gradient-btn {
            width: 100%;
        }
    }
</style>
</div>
