<?php

use Livewire\Component;

new class extends Component {
    public array $section = [];
    public array $items = [];
    public array $company = [];
    public array $social = [];
    public array $services = [];
    public bool $isWhiteBackground = false;
    public ?int $paddingTop = null;
    public ?int $paddingBottom = null;

    // Default static data for services
    private array $defaultItems = [
        [
            'icon' => 'lucide:shirt',
            'title' => 'Wash & Fold',
            'description' => 'Professional washing, drying, and folding service for your everyday clothes.',
            'price' => 'From $2.50/lb',
            'features' => [
                ['text' => 'Same-day service available'],
                ['text' => 'Eco-friendly detergents'],
                ['text' => 'Fabric softener included'],
            ],
            'button_text' => 'Learn More',
            'button_url' => '#contact',
            'button_icon' => 'lucide:arrow-right',
        ],
        [
            'icon' => 'lucide:sparkles',
            'title' => 'Dry Cleaning',
            'description' => 'Expert care for your delicate and special garments that need extra attention.',
            'price' => 'From $8.99/item',
            'features' => [
                ['text' => 'Suits & formal wear'],
                ['text' => 'Delicate fabrics'],
                ['text' => 'Stain treatment included'],
            ],
            'button_text' => 'Learn More',
            'button_url' => '#contact',
            'button_icon' => 'lucide:arrow-right',
        ],
        [
            'icon' => 'lucide:wind',
            'title' => 'Press & Iron',
            'description' => 'Professional pressing and ironing for a crisp, polished look.',
            'price' => 'From $3.50/item',
            'features' => [
                ['text' => 'Steam pressing'],
                ['text' => 'Hand finishing'],
                ['text' => 'Ready to wear'],
            ],
            'button_text' => 'Learn More',
            'button_url' => '#contact',
            'button_icon' => 'lucide:arrow-right',
        ],
        [
            'icon' => 'lucide:bed-double',
            'title' => 'Bedding & Linens',
            'description' => 'Fresh, clean bedding and household linens for maximum comfort.',
            'price' => 'From $15.99/set',
            'features' => [
                ['text' => 'Comforters & duvets'],
                ['text' => 'Sheets & pillowcases'],
                ['text' => 'Allergen removal'],
            ],
            'button_text' => 'Learn More',
            'button_url' => '#contact',
            'button_icon' => 'lucide:arrow-right',
        ],
        [
            'icon' => 'lucide:gem',
            'title' => 'Premium Care',
            'description' => 'Specialized treatment for luxury fabrics and designer pieces.',
            'price' => 'Custom pricing',
            'features' => [
                ['text' => 'Designer labels'],
                ['text' => 'Silk & cashmere'],
                ['text' => 'Hand wash only items'],
            ],
            'button_text' => 'Learn More',
            'button_url' => '#contact',
            'button_icon' => 'lucide:arrow-right',
        ],
        [
            'icon' => 'lucide:zap',
            'title' => 'Express Service',
            'description' => 'Urgent cleaning with quick turnaround when you need it fast.',
            'price' => '+50% rush fee',
            'features' => [
                ['text' => '2-hour turnaround'],
                ['text' => 'Priority handling'],
                ['text' => 'Delivery included'],
            ],
            'button_text' => 'Learn More',
            'button_url' => '#contact',
            'button_icon' => 'lucide:arrow-right',
        ],
    ];

    public function mount(
        array $section = [],
        array $items = [],
        array $company = [],
        array $social = [],
        array $services = [],
        bool $isWhiteBackground = false,
        ?int $paddingTop = null,
        ?int $paddingBottom = null
    ) {
        $this->company = $company;
        $this->social = $social;
        $this->isWhiteBackground = $isWhiteBackground;
        $this->paddingTop = $paddingTop;
        $this->paddingBottom = $paddingBottom;

        // Build section data with defaults
        $this->section = array_merge([
            'badge' => 'Our Services',
            'title' => 'Professional Laundry Services',
            'subtitle' => 'From everyday laundry to delicate dry cleaning, we\'ve got all your garment care needs covered.',
        ], $section);

        // Use items from section content (CMS), then $items prop, then default static data
        $this->items = !empty($section['items']) ? $section['items'] : (!empty($items) ? $items : $this->defaultItems);
    }

    /**
     * Get custom padding style if set
     */
    public function getCustomPaddingStyle(): string
    {
        $styles = [];
        if ($this->paddingTop !== null) {
            $styles[] = "padding-top: {$this->paddingTop}px";
        }
        if ($this->paddingBottom !== null) {
            $styles[] = "padding-bottom: {$this->paddingBottom}px";
        }
        return !empty($styles) ? implode('; ', $styles) : '';
    }
};
?>

<div>
<section id="services" class="services-horizontal-section {{ $isWhiteBackground ? 'services-horizontal-white' : 'services-horizontal-gray' }}" @if($this->getCustomPaddingStyle()) style="{{ $this->getCustomPaddingStyle() }}" @endif>
    {{-- Background Decoration --}}
    <div class="services-horizontal-bg">
        <div class="services-horizontal-circle services-horizontal-circle-1"></div>
        <div class="services-horizontal-circle services-horizontal-circle-2"></div>
        <div class="services-horizontal-line"></div>
    </div>

    <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 relative">
        {{-- Section Header --}}
        <div class="services-horizontal-header">
            <div class="services-horizontal-header-left">
                <span class="services-horizontal-badge">{{ $section['badge'] }}</span>
                <h2 class="services-horizontal-title">{{ $section['title'] }}</h2>
            </div>
            <div class="services-horizontal-header-right">
                <p class="services-horizontal-subtitle">{{ $section['subtitle'] }}</p>
            </div>
        </div>

        {{-- Services List --}}
        <div class="services-horizontal-list">
            @foreach ($items as $index => $item)
                <div class="services-horizontal-item scroll-animate" style="animation-delay: {{ $index * 100 }}ms">
                    {{-- Left: Icon & Number --}}
                    <div class="services-horizontal-icon-wrap">
                        <div class="services-horizontal-icon">
                            <x-icon icon="{{ $item['icon'] ?? 'heroicons:sparkles' }}" class="w-8 h-8" />
                        </div>
                        <div class="services-horizontal-connector"></div>
                    </div>

                    {{-- Center: Content --}}
                    <div class="services-horizontal-content">
                        <div class="services-horizontal-content-top">
                            <h3 class="services-horizontal-item-title">{{ $item['title'] ?? '' }}</h3>
                            @if (!empty($item['price']))
                                <span class="services-horizontal-price">{{ $item['price'] }}</span>
                            @endif
                        </div>
                        <p class="services-horizontal-desc">{{ $item['description'] ?? '' }}</p>

                        {{-- Features in a row --}}
                        @if (!empty($item['features']))
                            <div class="services-horizontal-features">
                                @foreach ($item['features'] as $feature)
                                    <span class="services-horizontal-feature">
                                        <x-icon icon="lucide:check" class="w-3.5 h-3.5" />
                                        <span>{{ is_array($feature) ? ($feature['text'] ?? '') : $feature }}</span>
                                    </span>
                                @endforeach
                            </div>
                        @endif
                    </div>

                    {{-- Right: Action --}}
                    @if (!empty($item['button_text']) && !empty($item['button_url']))
                        <div class="services-horizontal-action">
                            <a href="{{ $item['button_url'] }}" class="services-horizontal-btn">
                                <span>{{ $item['button_text'] }}</span>
                                <x-icon icon="{{ $item['button_icon'] ?? 'lucide:arrow-right' }}" class="w-4 h-4" />
                            </a>
                        </div>
                    @endif
                </div>
            @endforeach
        </div>
    </div>
</section>

<style>
    /* ===== Services Horizontal Section ===== */
    .services-horizontal-section {
        position: relative;
        padding: 6rem 0;
        overflow: hidden;
    }

    .services-horizontal-white {
        background: #ffffff;
    }

    .services-horizontal-gray {
        background: linear-gradient(180deg, #f8fafc 0%, #ffffff 100%);
    }

    .dark .services-horizontal-white {
        background: var(--color-bg-secondary);
    }

    .dark .services-horizontal-gray {
        background: linear-gradient(180deg, var(--color-bg-primary) 0%, var(--color-bg-secondary) 100%);
    }

    /* ===== Background ===== */
    .services-horizontal-bg {
        position: absolute;
        inset: 0;
        pointer-events: none;
        overflow: hidden;
    }

    .services-horizontal-circle {
        position: absolute;
        border-radius: 50%;
        border: 1px solid var(--color-primary-100);
    }

    .services-horizontal-circle-1 {
        width: 600px;
        height: 600px;
        top: -300px;
        right: -200px;
    }

    .services-horizontal-circle-2 {
        width: 400px;
        height: 400px;
        bottom: -200px;
        left: -150px;
    }

    .dark .services-horizontal-circle {
        border-color: var(--color-primary-900);
    }

    .services-horizontal-line {
        position: absolute;
        left: 50%;
        top: 0;
        bottom: 0;
        width: 1px;
        background: linear-gradient(180deg, transparent 0%, var(--color-primary-100) 10%, var(--color-primary-100) 90%, transparent 100%);
        opacity: 0.5;
    }

    .dark .services-horizontal-line {
        background: linear-gradient(180deg, transparent 0%, var(--color-primary-900) 10%, var(--color-primary-900) 90%, transparent 100%);
    }

    /* ===== Header ===== */
    .services-horizontal-header {
        display: flex;
        flex-direction: column;
        gap: 1.5rem;
        margin-bottom: 4rem;
    }

    @media (min-width: 768px) {
        .services-horizontal-header {
            flex-direction: row;
            align-items: flex-end;
            justify-content: space-between;
        }
    }

    .services-horizontal-header-left {
        flex: 1;
    }

    .services-horizontal-header-right {
        max-width: 400px;
    }

    .services-horizontal-badge {
        display: inline-block;
        padding: 0.375rem 1rem;
        background: linear-gradient(135deg, var(--color-primary-500), var(--color-secondary-500));
        color: white;
        font-size: 0.75rem;
        font-weight: 700;
        letter-spacing: 0.1em;
        text-transform: uppercase;
        border-radius: 9999px;
        margin-bottom: 1rem;
    }

    .services-horizontal-title {
        font-family: var(--font-heading);
        font-size: 2.5rem;
        font-weight: 800;
        color: var(--color-gray-900);
        line-height: 1.2;
    }

    .dark .services-horizontal-title {
        color: var(--color-text-primary);
    }

    .services-horizontal-subtitle {
        font-size: 1.0625rem;
        color: var(--color-gray-600);
        line-height: 1.7;
    }

    .dark .services-horizontal-subtitle {
        color: var(--color-text-tertiary);
    }

    /* ===== List ===== */
    .services-horizontal-list {
        display: flex;
        flex-direction: column;
        gap: 0;
    }

    /* ===== Item ===== */
    .services-horizontal-item {
        display: grid;
        grid-template-columns: auto 1fr auto;
        gap: 1.5rem;
        padding: 2rem 0;
        border-bottom: 1px solid var(--color-gray-100);
        transition: all 0.3s ease;
    }

    .services-horizontal-item:last-child {
        border-bottom: none;
    }

    .services-horizontal-item:hover {
        background: var(--color-gray-50);
        margin: 0 -1.5rem;
        padding-left: 1.5rem;
        padding-right: 1.5rem;
        border-radius: 1rem;
    }

    .dark .services-horizontal-item {
        border-bottom-color: var(--color-border);
    }

    .dark .services-horizontal-item:hover {
        background: var(--color-bg-tertiary);
    }

    /* ===== Icon Wrap ===== */
    .services-horizontal-icon-wrap {
        display: flex;
        flex-direction: column;
        align-items: center;
    }

    .services-horizontal-icon {
        width: 4rem;
        height: 4rem;
        display: flex;
        align-items: center;
        justify-content: center;
        background: white;
        border: 2px solid var(--color-primary-100);
        border-radius: 1rem;
        color: var(--color-primary-600);
        transition: all 0.3s ease;
        position: relative;
        z-index: 1;
    }

    .services-horizontal-item:hover .services-horizontal-icon {
        background: var(--color-primary-600);
        border-color: var(--color-primary-600);
        color: white;
        transform: scale(1.1) rotate(-5deg);
    }

    .dark .services-horizontal-icon {
        background: var(--color-card-bg);
        border-color: var(--color-primary-800);
        color: var(--color-primary-400);
    }

    .dark .services-horizontal-item:hover .services-horizontal-icon {
        background: var(--color-primary-500);
        border-color: var(--color-primary-500);
        color: white;
    }

    .services-horizontal-connector {
        width: 2px;
        flex: 1;
        background: linear-gradient(180deg, var(--color-primary-200), transparent);
        margin-top: 0.5rem;
    }

    .services-horizontal-item:last-child .services-horizontal-connector {
        display: none;
    }

    .dark .services-horizontal-connector {
        background: linear-gradient(180deg, var(--color-primary-800), transparent);
    }

    /* ===== Content ===== */
    .services-horizontal-content {
        flex: 1;
        display: flex;
        flex-direction: column;
        gap: 0.75rem;
    }

    .services-horizontal-content-top {
        display: flex;
        align-items: center;
        gap: 1rem;
        flex-wrap: wrap;
    }

    .services-horizontal-item-title {
        font-family: var(--font-heading);
        font-size: 1.375rem;
        font-weight: 700;
        color: var(--color-gray-900);
    }

    .dark .services-horizontal-item-title {
        color: var(--color-text-primary);
    }

    .services-horizontal-price {
        font-size: 0.875rem;
        font-weight: 600;
        color: var(--color-primary-600);
        padding: 0.25rem 0.75rem;
        background: var(--color-primary-50);
        border-radius: 9999px;
    }

    .dark .services-horizontal-price {
        color: var(--color-primary-400);
        background: rgba(var(--color-primary-rgb, 37, 99, 235), 0.15);
    }

    .services-horizontal-desc {
        font-size: 0.9375rem;
        color: var(--color-gray-600);
        line-height: 1.6;
        max-width: 600px;
    }

    .dark .services-horizontal-desc {
        color: var(--color-text-tertiary);
    }

    .services-horizontal-features {
        display: flex;
        flex-wrap: wrap;
        gap: 0.75rem;
        margin-top: 0.5rem;
    }

    .services-horizontal-feature {
        display: inline-flex;
        align-items: center;
        gap: 0.375rem;
        font-size: 0.8125rem;
        color: var(--color-gray-600);
        padding: 0.25rem 0.625rem;
        background: var(--color-gray-100);
        border-radius: 9999px;
    }

    .services-horizontal-feature svg {
        color: var(--color-primary-500);
    }

    .dark .services-horizontal-feature {
        color: var(--color-text-tertiary);
        background: var(--color-bg-tertiary);
    }

    .dark .services-horizontal-feature svg {
        color: var(--color-primary-400);
    }

    /* ===== Action ===== */
    .services-horizontal-action {
        display: flex;
        align-items: center;
    }

    .services-horizontal-btn {
        display: inline-flex;
        align-items: center;
        gap: 0.5rem;
        padding: 0.75rem 1.5rem;
        background: var(--color-gray-900);
        color: white;
        font-size: 0.875rem;
        font-weight: 600;
        border-radius: 0.75rem;
        transition: all 0.3s ease;
        white-space: nowrap;
    }

    .services-horizontal-btn:hover {
        background: var(--color-primary-600);
        transform: translateX(4px);
    }

    .dark .services-horizontal-btn {
        background: var(--color-primary-600);
    }

    .dark .services-horizontal-btn:hover {
        background: var(--color-primary-500);
    }

    /* ===== RTL Support ===== */
    [dir="rtl"] .services-horizontal-section {
        direction: rtl;
    }

    [dir="rtl"] .services-horizontal-header {
        text-align: right;
    }

    [dir="rtl"] .services-horizontal-header-left {
        text-align: right;
    }

    [dir="rtl"] .services-horizontal-header-right {
        text-align: right;
    }

    [dir="rtl"] .services-horizontal-item {
        direction: rtl;
    }

    [dir="rtl"] .services-horizontal-content {
        text-align: right;
    }

    [dir="rtl"] .services-horizontal-content-top {
        flex-direction: row-reverse;
        justify-content: flex-end;
    }

    [dir="rtl"] .services-horizontal-features {
        justify-content: flex-start;
    }

    [dir="rtl"] .services-horizontal-feature {
        flex-direction: row-reverse;
    }

    [dir="rtl"] .services-horizontal-btn {
        flex-direction: row-reverse;
    }

    [dir="rtl"] .services-horizontal-btn:hover {
        transform: translateX(-4px);
    }

    [dir="rtl"] .services-horizontal-btn svg {
        transform: scaleX(-1);
    }

    [dir="rtl"] .services-horizontal-item:hover .services-horizontal-icon {
        transform: scale(1.1) rotate(5deg);
    }

    [dir="rtl"] .services-horizontal-circle-1 {
        right: auto;
        left: -200px;
    }

    [dir="rtl"] .services-horizontal-circle-2 {
        left: auto;
        right: -150px;
    }

    [dir="rtl"] .services-horizontal-line {
        left: auto;
        right: 50%;
    }

    /* ===== Responsive ===== */
    @media (max-width: 768px) {
        .services-horizontal-section {
            padding: 4rem 0;
        }

        .services-horizontal-title {
            font-size: 2rem;
        }

        .services-horizontal-item {
            grid-template-columns: 1fr;
            gap: 1rem;
        }

        .services-horizontal-icon-wrap {
            flex-direction: row;
            gap: 1rem;
        }

        .services-horizontal-connector {
            width: 100%;
            height: 2px;
            margin-top: 0;
            margin-left: 0.5rem;
        }

        .services-horizontal-item:last-child .services-horizontal-connector {
            display: block;
            background: transparent;
        }

        .services-horizontal-action {
            margin-top: 0.5rem;
        }

        .services-horizontal-btn {
            width: 100%;
            justify-content: center;
        }

        .services-horizontal-item:hover {
            margin: 0 -1rem;
            padding-left: 1rem;
            padding-right: 1rem;
        }
    }
</style>
</div>
