<?php

use Livewire\Component;

new class extends Component {
    public array $section = [];
    public array $items = [];
    public array $company = [];
    public array $social = [];
    public array $services = [];
    public bool $isWhiteBackground = false;
    public ?int $paddingTop = null;
    public ?int $paddingBottom = null;

    // Default static data for services
    private array $defaultItems = [
        [
            'icon' => 'lucide:shirt',
            'title' => 'Wash & Fold',
            'description' => 'Professional washing, drying, and folding service for your everyday clothes.',
            'price' => 'From $2.50/lb',
            'features' => [
                ['text' => 'Same-day service available'],
                ['text' => 'Eco-friendly detergents'],
                ['text' => 'Fabric softener included'],
            ],
            'button_text' => 'Learn More',
            'button_url' => '#contact',
            'button_icon' => 'lucide:arrow-right',
        ],
        [
            'icon' => 'lucide:sparkles',
            'title' => 'Dry Cleaning',
            'description' => 'Expert care for your delicate and special garments that need extra attention.',
            'price' => 'From $8.99/item',
            'features' => [
                ['text' => 'Suits & formal wear'],
                ['text' => 'Delicate fabrics'],
                ['text' => 'Stain treatment included'],
            ],
            'button_text' => 'Learn More',
            'button_url' => '#contact',
            'button_icon' => 'lucide:arrow-right',
        ],
        [
            'icon' => 'lucide:wind',
            'title' => 'Press & Iron',
            'description' => 'Professional pressing and ironing for a crisp, polished look.',
            'price' => 'From $3.50/item',
            'features' => [
                ['text' => 'Steam pressing'],
                ['text' => 'Hand finishing'],
                ['text' => 'Ready to wear'],
            ],
            'button_text' => 'Learn More',
            'button_url' => '#contact',
            'button_icon' => 'lucide:arrow-right',
        ],
        [
            'icon' => 'lucide:bed-double',
            'title' => 'Bedding & Linens',
            'description' => 'Fresh, clean bedding and household linens for maximum comfort.',
            'price' => 'From $15.99/set',
            'features' => [
                ['text' => 'Comforters & duvets'],
                ['text' => 'Sheets & pillowcases'],
                ['text' => 'Allergen removal'],
            ],
            'button_text' => 'Learn More',
            'button_url' => '#contact',
            'button_icon' => 'lucide:arrow-right',
        ],
        [
            'icon' => 'lucide:gem',
            'title' => 'Premium Care',
            'description' => 'Specialized treatment for luxury fabrics and designer pieces.',
            'price' => 'Custom pricing',
            'features' => [
                ['text' => 'Designer labels'],
                ['text' => 'Silk & cashmere'],
                ['text' => 'Hand wash only items'],
            ],
            'button_text' => 'Learn More',
            'button_url' => '#contact',
            'button_icon' => 'lucide:arrow-right',
        ],
        [
            'icon' => 'lucide:zap',
            'title' => 'Express Service',
            'description' => 'Urgent cleaning with quick turnaround when you need it fast.',
            'price' => '+50% rush fee',
            'features' => [
                ['text' => '2-hour turnaround'],
                ['text' => 'Priority handling'],
                ['text' => 'Delivery included'],
            ],
            'button_text' => 'Learn More',
            'button_url' => '#contact',
            'button_icon' => 'lucide:arrow-right',
        ],
    ];

    public function mount(
        array $section = [],
        array $items = [],
        array $company = [],
        array $social = [],
        array $services = [],
        bool $isWhiteBackground = false,
        ?int $paddingTop = null,
        ?int $paddingBottom = null
    ) {
        $this->company = $company;
        $this->social = $social;
        $this->isWhiteBackground = $isWhiteBackground;
        $this->paddingTop = $paddingTop;
        $this->paddingBottom = $paddingBottom;

        // Build section data with defaults
        $this->section = array_merge([
            'badge' => 'Our Services',
            'title' => 'Professional Laundry Services',
            'subtitle' => 'From everyday laundry to delicate dry cleaning, we\'ve got all your garment care needs covered.',
        ], $section);

        // Use items from section content (CMS), then $items prop, then default static data
        $this->items = !empty($section['items']) ? $section['items'] : (!empty($items) ? $items : $this->defaultItems);
    }

    /**
     * Get custom padding style if set
     */
    public function getCustomPaddingStyle(): string
    {
        $styles = [];
        if ($this->paddingTop !== null) {
            $styles[] = "padding-top: {$this->paddingTop}px";
        }
        if ($this->paddingBottom !== null) {
            $styles[] = "padding-bottom: {$this->paddingBottom}px";
        }
        return !empty($styles) ? implode('; ', $styles) : '';
    }
};
?>

<div>
<section id="services" class="services-icons-section {{ $isWhiteBackground ? 'services-icons-white' : 'services-icons-gray' }}" @if($this->getCustomPaddingStyle()) style="{{ $this->getCustomPaddingStyle() }}" @endif>
    {{-- Background Elements --}}
    <div class="services-icons-bg">
        <div class="services-icons-hex services-icons-hex-1"></div>
        <div class="services-icons-hex services-icons-hex-2"></div>
        <div class="services-icons-hex services-icons-hex-3"></div>
    </div>

    <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 relative">
        {{-- Section Header --}}
        <div class="services-icons-header">
            <span class="services-icons-badge">
                <x-icon icon="lucide:sparkles" class="w-4 h-4" />
                {{ $section['badge'] }}
            </span>
            <h2 class="services-icons-title">{{ $section['title'] }}</h2>
            <p class="services-icons-subtitle">{{ $section['subtitle'] }}</p>
        </div>

        {{-- Services Grid --}}
        <div class="services-icons-grid">
            @foreach ($items as $index => $item)
                <div class="services-icons-item scroll-animate" style="animation-delay: {{ $index * 100 }}ms">
                    {{-- Large Icon --}}
                    <div class="services-icons-icon-wrap">
                        <div class="services-icons-icon">
                            <x-icon icon="{{ $item['icon'] ?? 'heroicons:sparkles' }}" class="w-10 h-10" />
                        </div>
                        <div class="services-icons-icon-ring"></div>
                        <div class="services-icons-icon-dots"></div>
                    </div>

                    {{-- Content --}}
                    <div class="services-icons-content">
                        <h3 class="services-icons-item-title">{{ $item['title'] ?? '' }}</h3>
                        <p class="services-icons-desc">{{ $item['description'] ?? '' }}</p>

                        {{-- Price --}}
                        @if (!empty($item['price']))
                            <div class="services-icons-price">
                                <x-icon icon="lucide:tag" class="w-4 h-4" />
                                <span>{{ $item['price'] }}</span>
                            </div>
                        @endif

                        {{-- Features --}}
                        @if (!empty($item['features']))
                            <div class="services-icons-features">
                                @foreach ($item['features'] as $feature)
                                    <div class="services-icons-feature">
                                        <span class="services-icons-feature-dot"></span>
                                        <span>{{ is_array($feature) ? ($feature['text'] ?? '') : $feature }}</span>
                                    </div>
                                @endforeach
                            </div>
                        @endif

                        {{-- Button --}}
                        @if (!empty($item['button_text']) && !empty($item['button_url']))
                            <a href="{{ $item['button_url'] }}" class="services-icons-btn">
                                {{ $item['button_text'] }}
                                <x-icon icon="{{ $item['button_icon'] ?? 'lucide:arrow-up-right' }}" class="w-4 h-4" />
                            </a>
                        @endif
                    </div>
                </div>
            @endforeach
        </div>
    </div>
</section>

<style>
    /* ===== Services Icons Section ===== */
    .services-icons-section {
        position: relative;
        padding: 6rem 0;
        overflow: hidden;
    }

    .services-icons-white {
        background: #ffffff;
    }

    .services-icons-gray {
        background: #f9fafb;
    }

    .dark .services-icons-white {
        background: var(--color-bg-secondary);
    }

    .dark .services-icons-gray {
        background: var(--color-bg-primary);
    }

    /* ===== Background ===== */
    .services-icons-bg {
        position: absolute;
        inset: 0;
        pointer-events: none;
        overflow: hidden;
    }

    .services-icons-hex {
        position: absolute;
        width: 200px;
        height: 200px;
        background: var(--color-primary-100);
        clip-path: polygon(50% 0%, 100% 25%, 100% 75%, 50% 100%, 0% 75%, 0% 25%);
        opacity: 0.3;
    }

    .services-icons-hex-1 {
        top: 10%;
        right: 5%;
        transform: rotate(15deg);
    }

    .services-icons-hex-2 {
        bottom: 20%;
        left: 3%;
        width: 150px;
        height: 150px;
        transform: rotate(-10deg);
    }

    .services-icons-hex-3 {
        top: 50%;
        right: 15%;
        width: 100px;
        height: 100px;
        background: var(--color-secondary-100);
        transform: rotate(30deg);
    }

    .dark .services-icons-hex {
        background: var(--color-primary-900);
        opacity: 0.15;
    }

    .dark .services-icons-hex-3 {
        background: var(--color-secondary-900);
    }

    /* ===== Header ===== */
    .services-icons-header {
        text-align: center;
        margin-bottom: 4rem;
    }

    .services-icons-badge {
        display: inline-flex;
        align-items: center;
        gap: 0.5rem;
        padding: 0.5rem 1.25rem;
        background: var(--color-primary-600);
        color: white;
        font-size: 0.8125rem;
        font-weight: 600;
        border-radius: 9999px;
        margin-bottom: 1.5rem;
    }

    .services-icons-title {
        font-family: var(--font-heading);
        font-size: 2.5rem;
        font-weight: 800;
        color: var(--color-gray-900);
        margin-bottom: 1rem;
    }

    .dark .services-icons-title {
        color: var(--color-text-primary);
    }

    .services-icons-subtitle {
        font-size: 1.125rem;
        color: var(--color-gray-600);
        max-width: 600px;
        margin: 0 auto;
    }

    .dark .services-icons-subtitle {
        color: var(--color-text-tertiary);
    }

    /* ===== Grid ===== */
    .services-icons-grid {
        display: grid;
        gap: 2rem;
    }

    @media (min-width: 640px) {
        .services-icons-grid {
            grid-template-columns: repeat(2, 1fr);
        }
    }

    @media (min-width: 1024px) {
        .services-icons-grid {
            grid-template-columns: repeat(3, 1fr);
        }
    }

    /* ===== Item ===== */
    .services-icons-item {
        position: relative;
        text-align: center;
        padding: 2.5rem 1.5rem;
        background: white;
        border: 1px solid var(--color-gray-200);
        border-radius: 1.5rem;
        transition: all 0.4s cubic-bezier(0.4, 0, 0.2, 1);
    }

    .services-icons-item:hover {
        transform: translateY(-8px);
        border-color: var(--color-primary-200);
        box-shadow: 0 20px 50px -15px rgba(0, 0, 0, 0.1);
    }

    .dark .services-icons-item {
        background: var(--color-card-bg);
        border-color: var(--color-border);
    }

    .dark .services-icons-item:hover {
        border-color: var(--color-primary-700);
        box-shadow: 0 20px 50px -15px rgba(0, 0, 0, 0.4);
    }

    /* ===== Icon Wrap ===== */
    .services-icons-icon-wrap {
        position: relative;
        display: inline-block;
        margin-bottom: 1.5rem;
    }

    .services-icons-icon {
        position: relative;
        width: 5rem;
        height: 5rem;
        display: flex;
        align-items: center;
        justify-content: center;
        background: linear-gradient(135deg, var(--color-primary-500), var(--color-primary-600));
        border-radius: 1.25rem;
        color: white;
        transition: all 0.4s ease;
        z-index: 2;
    }

    .services-icons-item:hover .services-icons-icon {
        transform: rotate(-10deg) scale(1.1);
        border-radius: 1.5rem;
    }

    .services-icons-icon-ring {
        position: absolute;
        inset: -8px;
        border: 2px dashed var(--color-primary-200);
        border-radius: 1.5rem;
        opacity: 0;
        transition: all 0.4s ease;
    }

    .services-icons-item:hover .services-icons-icon-ring {
        opacity: 1;
        animation: services-icons-spin 10s linear infinite;
    }

    @keyframes services-icons-spin {
        from { transform: rotate(0deg); }
        to { transform: rotate(360deg); }
    }

    .dark .services-icons-icon-ring {
        border-color: var(--color-primary-700);
    }

    .services-icons-icon-dots {
        position: absolute;
        width: 8px;
        height: 8px;
        background: var(--color-secondary-400);
        border-radius: 50%;
        top: -4px;
        right: -4px;
        z-index: 3;
    }

    .services-icons-icon-dots::before {
        content: '';
        position: absolute;
        width: 6px;
        height: 6px;
        background: var(--color-primary-300);
        border-radius: 50%;
        bottom: -40px;
        left: -40px;
    }

    /* ===== Content ===== */
    .services-icons-content {
        display: flex;
        flex-direction: column;
        align-items: center;
    }

    .services-icons-item-title {
        font-family: var(--font-heading);
        font-size: 1.375rem;
        font-weight: 700;
        color: var(--color-gray-900);
        margin-bottom: 0.625rem;
    }

    .dark .services-icons-item-title {
        color: var(--color-text-primary);
    }

    .services-icons-desc {
        font-size: 0.9375rem;
        color: var(--color-gray-600);
        line-height: 1.6;
        margin-bottom: 1rem;
    }

    .dark .services-icons-desc {
        color: var(--color-text-tertiary);
    }

    /* ===== Price ===== */
    .services-icons-price {
        display: inline-flex;
        align-items: center;
        gap: 0.375rem;
        padding: 0.5rem 1rem;
        background: linear-gradient(135deg, var(--color-primary-50), var(--color-secondary-50));
        border: 1px solid var(--color-primary-100);
        border-radius: 9999px;
        font-size: 0.875rem;
        font-weight: 700;
        color: var(--color-primary-700);
        margin-bottom: 1.25rem;
    }

    .dark .services-icons-price {
        background: linear-gradient(135deg, rgba(var(--color-primary-rgb, 37, 99, 235), 0.1), rgba(var(--color-secondary-rgb, 124, 58, 237), 0.1));
        border-color: var(--color-primary-800);
        color: var(--color-primary-400);
    }

    /* ===== Features ===== */
    .services-icons-features {
        display: flex;
        flex-direction: column;
        gap: 0.5rem;
        margin-bottom: 1.25rem;
        width: 100%;
    }

    .services-icons-feature {
        display: flex;
        align-items: center;
        justify-content: center;
        gap: 0.5rem;
        font-size: 0.875rem;
        color: var(--color-gray-600);
    }

    .dark .services-icons-feature {
        color: var(--color-text-tertiary);
    }

    .services-icons-feature-dot {
        width: 6px;
        height: 6px;
        background: var(--color-primary-400);
        border-radius: 50%;
        flex-shrink: 0;
    }

    .dark .services-icons-feature-dot {
        background: var(--color-primary-500);
    }

    /* ===== Button ===== */
    .services-icons-btn {
        display: inline-flex;
        align-items: center;
        gap: 0.5rem;
        padding: 0.75rem 1.5rem;
        background: var(--color-gray-900);
        color: white;
        font-size: 0.875rem;
        font-weight: 600;
        border-radius: 0.75rem;
        transition: all 0.3s ease;
    }

    .services-icons-btn:hover {
        background: var(--color-primary-600);
        transform: translateY(-2px);
    }

    .dark .services-icons-btn {
        background: var(--color-primary-600);
    }

    .dark .services-icons-btn:hover {
        background: var(--color-primary-500);
    }

    /* ===== RTL Support ===== */
    [dir="rtl"] .services-icons-section {
        direction: rtl;
    }

    [dir="rtl"] .services-icons-badge {
        flex-direction: row-reverse;
    }

    [dir="rtl"] .services-icons-price {
        flex-direction: row-reverse;
    }

    [dir="rtl"] .services-icons-feature {
        flex-direction: row-reverse;
    }

    [dir="rtl"] .services-icons-btn {
        flex-direction: row-reverse;
    }

    [dir="rtl"] .services-icons-btn svg {
        transform: scaleX(-1) rotate(-45deg);
    }

    [dir="rtl"] .services-icons-item:hover .services-icons-icon {
        transform: rotate(10deg) scale(1.1);
    }

    [dir="rtl"] .services-icons-hex-1 {
        right: auto;
        left: 5%;
        transform: rotate(-15deg);
    }

    [dir="rtl"] .services-icons-hex-2 {
        left: auto;
        right: 3%;
        transform: rotate(10deg);
    }

    [dir="rtl"] .services-icons-hex-3 {
        right: auto;
        left: 15%;
        transform: rotate(-30deg);
    }

    [dir="rtl"] .services-icons-icon-dots {
        right: auto;
        left: -4px;
    }

    [dir="rtl"] .services-icons-icon-dots::before {
        left: auto;
        right: -40px;
    }

    /* ===== Responsive ===== */
    @media (max-width: 640px) {
        .services-icons-section {
            padding: 4rem 0;
        }

        .services-icons-title {
            font-size: 2rem;
        }

        .services-icons-item {
            padding: 2rem 1.25rem;
        }

        .services-icons-icon {
            width: 4rem;
            height: 4rem;
        }

        .services-icons-icon svg {
            width: 2rem;
            height: 2rem;
        }

        .services-icons-btn {
            width: 100%;
            justify-content: center;
        }
    }
</style>
</div>
