<?php

use Livewire\Component;

new class extends Component {
    public array $section = [];
    public array $items = [];
    public array $company = [];
    public array $social = [];
    public array $services = [];
    public bool $isWhiteBackground = false;
    public ?int $paddingTop = null;
    public ?int $paddingBottom = null;

    // Default static data for services
    private array $defaultItems = [
        [
            'icon' => 'lucide:shirt',
            'title' => 'Wash & Fold',
            'description' => 'Professional washing, drying, and folding service for your everyday clothes.',
            'price' => 'From $2.50/lb',
            'features' => [
                ['text' => 'Same-day service available'],
                ['text' => 'Eco-friendly detergents'],
                ['text' => 'Fabric softener included'],
            ],
            'button_text' => 'Learn More',
            'button_url' => '#contact',
            'button_icon' => 'lucide:arrow-right',
        ],
        [
            'icon' => 'lucide:sparkles',
            'title' => 'Dry Cleaning',
            'description' => 'Expert care for your delicate and special garments that need extra attention.',
            'price' => 'From $8.99/item',
            'features' => [
                ['text' => 'Suits & formal wear'],
                ['text' => 'Delicate fabrics'],
                ['text' => 'Stain treatment included'],
            ],
            'button_text' => 'Learn More',
            'button_url' => '#contact',
            'button_icon' => 'lucide:arrow-right',
        ],
        [
            'icon' => 'lucide:wind',
            'title' => 'Press & Iron',
            'description' => 'Professional pressing and ironing for a crisp, polished look.',
            'price' => 'From $3.50/item',
            'features' => [
                ['text' => 'Steam pressing'],
                ['text' => 'Hand finishing'],
                ['text' => 'Ready to wear'],
            ],
            'button_text' => 'Learn More',
            'button_url' => '#contact',
            'button_icon' => 'lucide:arrow-right',
        ],
        [
            'icon' => 'lucide:bed-double',
            'title' => 'Bedding & Linens',
            'description' => 'Fresh, clean bedding and household linens for maximum comfort.',
            'price' => 'From $15.99/set',
            'features' => [
                ['text' => 'Comforters & duvets'],
                ['text' => 'Sheets & pillowcases'],
                ['text' => 'Allergen removal'],
            ],
            'button_text' => 'Learn More',
            'button_url' => '#contact',
            'button_icon' => 'lucide:arrow-right',
        ],
        [
            'icon' => 'lucide:gem',
            'title' => 'Premium Care',
            'description' => 'Specialized treatment for luxury fabrics and designer pieces.',
            'price' => 'Custom pricing',
            'features' => [
                ['text' => 'Designer labels'],
                ['text' => 'Silk & cashmere'],
                ['text' => 'Hand wash only items'],
            ],
            'button_text' => 'Learn More',
            'button_url' => '#contact',
            'button_icon' => 'lucide:arrow-right',
        ],
        [
            'icon' => 'lucide:zap',
            'title' => 'Express Service',
            'description' => 'Urgent cleaning with quick turnaround when you need it fast.',
            'price' => '+50% rush fee',
            'features' => [
                ['text' => '2-hour turnaround'],
                ['text' => 'Priority handling'],
                ['text' => 'Delivery included'],
            ],
            'button_text' => 'Learn More',
            'button_url' => '#contact',
            'button_icon' => 'lucide:arrow-right',
        ],
    ];

    public function mount(
        array $section = [],
        array $items = [],
        array $company = [],
        array $social = [],
        array $services = [],
        bool $isWhiteBackground = false,
        ?int $paddingTop = null,
        ?int $paddingBottom = null
    ) {
        $this->company = $company;
        $this->social = $social;
        $this->isWhiteBackground = $isWhiteBackground;
        $this->paddingTop = $paddingTop;
        $this->paddingBottom = $paddingBottom;

        // Build section data with defaults
        $this->section = array_merge([
            'badge' => 'Our Services',
            'title' => 'Professional Laundry Services',
            'subtitle' => 'From everyday laundry to delicate dry cleaning, we\'ve got all your garment care needs covered.',
        ], $section);

        // Use items from section content (CMS), then $items prop, then default static data
        $this->items = !empty($section['items']) ? $section['items'] : (!empty($items) ? $items : $this->defaultItems);
    }

    /**
     * Get custom padding style if set
     */
    public function getCustomPaddingStyle(): string
    {
        $styles = [];
        if ($this->paddingTop !== null) {
            $styles[] = "padding-top: {$this->paddingTop}px";
        }
        if ($this->paddingBottom !== null) {
            $styles[] = "padding-bottom: {$this->paddingBottom}px";
        }
        return !empty($styles) ? implode('; ', $styles) : '';
    }
};
?>

<div>
<section id="services" class="services-minimal-section {{ $isWhiteBackground ? 'services-minimal-white' : 'services-minimal-gray' }}" @if($this->getCustomPaddingStyle()) style="{{ $this->getCustomPaddingStyle() }}" @endif>
    <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        {{-- Section Header --}}
        <div class="services-minimal-header">
            <span class="services-minimal-badge">{{ $section['badge'] }}</span>
            <h2 class="services-minimal-title">{{ $section['title'] }}</h2>
            <p class="services-minimal-subtitle">{{ $section['subtitle'] }}</p>
            <div class="services-minimal-divider"></div>
        </div>

        {{-- Services Grid --}}
        <div class="services-minimal-grid">
            @foreach ($items as $index => $item)
                <div class="services-minimal-item scroll-animate" style="animation-delay: {{ $index * 100 }}ms">
                    {{-- Left Side - Number & Icon --}}
                    <div class="services-minimal-left">
                        <span class="services-minimal-number">{{ str_pad($index + 1, 2, '0', STR_PAD_LEFT) }}</span>
                        <div class="services-minimal-icon">
                            <x-icon icon="{{ $item['icon'] ?? 'heroicons:sparkles' }}" class="w-6 h-6" />
                        </div>
                    </div>

                    {{-- Right Side - Content --}}
                    <div class="services-minimal-right">
                        <div class="services-minimal-content">
                            <h3 class="services-minimal-item-title">{{ $item['title'] ?? '' }}</h3>
                            <p class="services-minimal-item-desc">{{ $item['description'] ?? '' }}</p>
                        </div>

                        @if (!empty($item['price']))
                            <div class="services-minimal-price">{{ $item['price'] }}</div>
                        @endif

                        @if (!empty($item['button_text']) && !empty($item['button_url']))
                            <a href="{{ $item['button_url'] }}" class="services-minimal-link">
                                <span>{{ $item['button_text'] }}</span>
                                <x-icon icon="{{ $item['button_icon'] ?? 'lucide:arrow-right' }}" class="w-4 h-4" />
                            </a>
                        @endif
                    </div>
                </div>
            @endforeach
        </div>
    </div>
</section>

<style>
    /* ===== Services Minimal Section ===== */
    .services-minimal-section {
        padding: 6rem 0;
    }

    .services-minimal-white {
        background: #ffffff;
    }

    .services-minimal-gray {
        background: #fafafa;
    }

    .dark .services-minimal-white {
        background: var(--color-bg-secondary);
    }

    .dark .services-minimal-gray {
        background: var(--color-bg-primary);
    }

    /* ===== Header ===== */
    .services-minimal-header {
        text-align: center;
        margin-bottom: 4rem;
    }

    .services-minimal-badge {
        display: inline-block;
        font-size: 0.75rem;
        font-weight: 700;
        letter-spacing: 0.15em;
        text-transform: uppercase;
        color: var(--color-primary-600);
        margin-bottom: 1rem;
    }

    .dark .services-minimal-badge {
        color: var(--color-primary-400);
    }

    .services-minimal-title {
        font-family: var(--font-heading);
        font-size: 2.5rem;
        font-weight: 700;
        color: var(--color-gray-900);
        margin-bottom: 1rem;
        letter-spacing: -0.025em;
    }

    .dark .services-minimal-title {
        color: var(--color-text-primary);
    }

    .services-minimal-subtitle {
        font-size: 1.125rem;
        color: var(--color-gray-500);
        max-width: 500px;
        margin: 0 auto 2rem;
    }

    .dark .services-minimal-subtitle {
        color: var(--color-text-tertiary);
    }

    .services-minimal-divider {
        width: 60px;
        height: 3px;
        background: var(--color-primary-500);
        margin: 0 auto;
    }

    /* ===== Grid ===== */
    .services-minimal-grid {
        display: flex;
        flex-direction: column;
        gap: 0;
    }

    /* ===== Item ===== */
    .services-minimal-item {
        display: flex;
        gap: 2rem;
        padding: 2rem 0;
        border-bottom: 1px solid var(--color-gray-200);
        transition: all 0.3s ease;
    }

    .services-minimal-item:last-child {
        border-bottom: none;
    }

    .services-minimal-item:hover {
        padding-left: 1rem;
        padding-right: 1rem;
        background: var(--color-gray-50);
        margin: 0 -1rem;
        border-radius: 0.75rem;
    }

    .dark .services-minimal-item {
        border-bottom-color: var(--color-border);
    }

    .dark .services-minimal-item:hover {
        background: var(--color-bg-tertiary);
    }

    /* ===== Left Side ===== */
    .services-minimal-left {
        display: flex;
        flex-direction: column;
        align-items: center;
        gap: 1rem;
        min-width: 60px;
    }

    .services-minimal-number {
        font-family: var(--font-heading);
        font-size: 0.875rem;
        font-weight: 700;
        color: var(--color-gray-300);
    }

    .dark .services-minimal-number {
        color: var(--color-gray-600);
    }

    .services-minimal-icon {
        width: 3rem;
        height: 3rem;
        display: flex;
        align-items: center;
        justify-content: center;
        border-radius: 50%;
        border: 2px solid var(--color-primary-200);
        color: var(--color-primary-600);
        transition: all 0.3s ease;
    }

    .services-minimal-item:hover .services-minimal-icon {
        background: var(--color-primary-600);
        border-color: var(--color-primary-600);
        color: white;
        transform: scale(1.1);
    }

    .dark .services-minimal-icon {
        border-color: var(--color-primary-700);
        color: var(--color-primary-400);
    }

    .dark .services-minimal-item:hover .services-minimal-icon {
        background: var(--color-primary-500);
        border-color: var(--color-primary-500);
        color: white;
    }

    /* ===== Right Side ===== */
    .services-minimal-right {
        flex: 1;
        display: flex;
        align-items: center;
        gap: 2rem;
        flex-wrap: wrap;
    }

    .services-minimal-content {
        flex: 1;
        min-width: 200px;
    }

    .services-minimal-item-title {
        font-family: var(--font-heading);
        font-size: 1.25rem;
        font-weight: 600;
        color: var(--color-gray-900);
        margin-bottom: 0.375rem;
    }

    .dark .services-minimal-item-title {
        color: var(--color-text-primary);
    }

    .services-minimal-item-desc {
        font-size: 0.9375rem;
        color: var(--color-gray-500);
        line-height: 1.6;
    }

    .dark .services-minimal-item-desc {
        color: var(--color-text-tertiary);
    }

    .services-minimal-price {
        font-family: var(--font-heading);
        font-size: 1rem;
        font-weight: 600;
        color: var(--color-primary-600);
        white-space: nowrap;
    }

    .dark .services-minimal-price {
        color: var(--color-primary-400);
    }

    .services-minimal-link {
        display: inline-flex;
        align-items: center;
        gap: 0.5rem;
        font-size: 0.875rem;
        font-weight: 600;
        color: var(--color-gray-900);
        transition: all 0.3s ease;
        white-space: nowrap;
    }

    .services-minimal-link:hover {
        color: var(--color-primary-600);
        gap: 0.75rem;
    }

    .dark .services-minimal-link {
        color: var(--color-text-primary);
    }

    .dark .services-minimal-link:hover {
        color: var(--color-primary-400);
    }

    /* ===== RTL Support ===== */
    [dir="rtl"] .services-minimal-section {
        direction: rtl;
    }

    [dir="rtl"] .services-minimal-item {
        flex-direction: row-reverse;
    }

    [dir="rtl"] .services-minimal-item:hover {
        padding-left: 1rem;
        padding-right: 1rem;
    }

    [dir="rtl"] .services-minimal-left {
        align-items: center;
    }

    [dir="rtl"] .services-minimal-right {
        text-align: right;
    }

    [dir="rtl"] .services-minimal-content {
        text-align: right;
    }

    [dir="rtl"] .services-minimal-link {
        flex-direction: row-reverse;
    }

    [dir="rtl"] .services-minimal-link svg {
        transform: scaleX(-1);
    }

    [dir="rtl"] .services-minimal-link:hover {
        gap: 0.75rem;
    }

    @media (max-width: 768px) {
        [dir="rtl"] .services-minimal-item {
            flex-direction: column;
        }

        [dir="rtl"] .services-minimal-left {
            flex-direction: row-reverse;
            justify-content: flex-end;
        }

        [dir="rtl"] .services-minimal-right {
            align-items: flex-end;
        }
    }

    /* ===== Responsive ===== */
    @media (max-width: 768px) {
        .services-minimal-section {
            padding: 4rem 0;
        }

        .services-minimal-title {
            font-size: 2rem;
        }

        .services-minimal-item {
            flex-direction: column;
            gap: 1rem;
        }

        .services-minimal-left {
            flex-direction: row;
            justify-content: flex-start;
        }

        .services-minimal-right {
            flex-direction: column;
            align-items: flex-start;
            gap: 1rem;
        }

        .services-minimal-item:hover {
            margin: 0;
            padding-left: 1rem;
            padding-right: 1rem;
        }
    }
</style>
</div>
