<?php

use Livewire\Component;

new class extends Component {
    public array $section = [];
    public array $items = [];
    public array $company = [];
    public array $social = [];
    public array $services = [];
    public bool $isWhiteBackground = false;
    public ?int $paddingTop = null;
    public ?int $paddingBottom = null;

    private array $defaultItems = [
        [
            'icon' => 'lucide:shirt',
            'title' => 'Wash & Fold',
            'description' => 'Professional washing, drying, and folding service for your everyday clothes.',
            'price' => 'From $2.50/lb',
            'features' => [
                ['text' => 'Same-day service available'],
                ['text' => 'Eco-friendly detergents'],
                ['text' => 'Fabric softener included'],
            ],
            'button_text' => 'Learn More',
            'button_url' => '#contact',
            'button_icon' => 'lucide:arrow-right',
        ],
        [
            'icon' => 'lucide:sparkles',
            'title' => 'Dry Cleaning',
            'description' => 'Expert care for your delicate and special garments that need extra attention.',
            'price' => 'From $8.99/item',
            'features' => [
                ['text' => 'Suits & formal wear'],
                ['text' => 'Delicate fabrics'],
                ['text' => 'Stain treatment included'],
            ],
            'button_text' => 'Learn More',
            'button_url' => '#contact',
            'button_icon' => 'lucide:arrow-right',
        ],
        [
            'icon' => 'lucide:wind',
            'title' => 'Press & Iron',
            'description' => 'Professional pressing and ironing for a crisp, polished look.',
            'price' => 'From $3.50/item',
            'features' => [
                ['text' => 'Steam pressing'],
                ['text' => 'Hand finishing'],
                ['text' => 'Ready to wear'],
            ],
            'button_text' => 'Learn More',
            'button_url' => '#contact',
            'button_icon' => 'lucide:arrow-right',
        ],
        [
            'icon' => 'lucide:bed-double',
            'title' => 'Bedding & Linens',
            'description' => 'Fresh, clean bedding and household linens for maximum comfort.',
            'price' => 'From $15.99/set',
            'features' => [
                ['text' => 'Comforters & duvets'],
                ['text' => 'Sheets & pillowcases'],
                ['text' => 'Allergen removal'],
            ],
            'button_text' => 'Learn More',
            'button_url' => '#contact',
            'button_icon' => 'lucide:arrow-right',
        ],
        [
            'icon' => 'lucide:gem',
            'title' => 'Premium Care',
            'description' => 'Specialized treatment for luxury fabrics and designer pieces.',
            'price' => 'Custom pricing',
            'features' => [
                ['text' => 'Designer labels'],
                ['text' => 'Silk & cashmere'],
                ['text' => 'Hand wash only items'],
            ],
            'button_text' => 'Learn More',
            'button_url' => '#contact',
            'button_icon' => 'lucide:arrow-right',
        ],
        [
            'icon' => 'lucide:zap',
            'title' => 'Express Service',
            'description' => 'Urgent cleaning with quick turnaround when you need it fast.',
            'price' => '+50% rush fee',
            'features' => [
                ['text' => '2-hour turnaround'],
                ['text' => 'Priority handling'],
                ['text' => 'Delivery included'],
            ],
            'button_text' => 'Learn More',
            'button_url' => '#contact',
            'button_icon' => 'lucide:arrow-right',
        ],
    ];

    public function mount(
        array $section = [],
        array $items = [],
        array $company = [],
        array $social = [],
        array $services = [],
        bool $isWhiteBackground = false,
        ?int $paddingTop = null,
        ?int $paddingBottom = null
    ) {
        $this->company = $company;
        $this->social = $social;
        $this->isWhiteBackground = $isWhiteBackground;
        $this->paddingTop = $paddingTop;
        $this->paddingBottom = $paddingBottom;

        $this->section = array_merge([
            'badge' => 'Our Services',
            'title' => 'Professional Laundry Services',
            'subtitle' => 'From everyday laundry to delicate dry cleaning, we\'ve got all your garment care needs covered.',
        ], $section);

        $this->items = !empty($section['items']) ? $section['items'] : (!empty($items) ? $items : $this->defaultItems);
    }

    public function getCustomPaddingStyle(): string
    {
        $styles = [];
        if ($this->paddingTop !== null) {
            $styles[] = "padding-top: {$this->paddingTop}px";
        }
        if ($this->paddingBottom !== null) {
            $styles[] = "padding-bottom: {$this->paddingBottom}px";
        }
        return !empty($styles) ? implode('; ', $styles) : '';
    }
};
?>

<div>
<section id="services" class="services-orbit-section {{ $isWhiteBackground ? 'services-orbit-white' : 'services-orbit-gray' }}" @if($this->getCustomPaddingStyle()) style="{{ $this->getCustomPaddingStyle() }}" @endif>
    {{-- Background Elements --}}
    <div class="services-orbit-bg">
        <div class="services-orbit-circle services-orbit-circle-1"></div>
        <div class="services-orbit-circle services-orbit-circle-2"></div>
        <div class="services-orbit-circle services-orbit-circle-3"></div>
        <div class="services-orbit-particles">
            @for ($i = 0; $i < 20; $i++)
                <div class="services-orbit-particle" style="--delay: {{ $i * 0.5 }}s; --x: {{ rand(0, 100) }}%; --y: {{ rand(0, 100) }}%;"></div>
            @endfor
        </div>
    </div>

    <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 relative">
        {{-- Section Header --}}
        <div class="services-orbit-header">
            <div class="services-orbit-badge">
                <div class="services-orbit-badge-orbit">
                    <span></span>
                </div>
                {{ $section['badge'] }}
            </div>
            <h2 class="services-orbit-title">{{ $section['title'] }}</h2>
            <p class="services-orbit-subtitle">{{ $section['subtitle'] }}</p>
        </div>

        {{-- Central Hub + Orbiting Cards --}}
        <div class="services-orbit-container">
            {{-- Central Hub --}}
            <div class="services-orbit-hub">
                <div class="services-orbit-hub-inner">
                    <div class="services-orbit-hub-icon">
                        <x-icon icon="lucide:sparkles" class="w-10 h-10" />
                    </div>
                    <div class="services-orbit-hub-text">
                        <span class="services-orbit-hub-count">{{ count($items) }}</span>
                        <span class="services-orbit-hub-label">Services</span>
                    </div>
                </div>
                <div class="services-orbit-hub-ring services-orbit-hub-ring-1"></div>
                <div class="services-orbit-hub-ring services-orbit-hub-ring-2"></div>
            </div>

            {{-- Service Cards --}}
            <div class="services-orbit-cards">
                @foreach ($items as $index => $item)
                    <div class="services-orbit-card scroll-animate" style="animation-delay: {{ $index * 100 }}ms">
                        <div class="services-orbit-card-connector"></div>
                        <div class="services-orbit-card-inner">
                            {{-- Card Header --}}
                            <div class="services-orbit-card-header">
                                <div class="services-orbit-icon">
                                    <x-icon icon="{{ $item['icon'] ?? 'heroicons:sparkles' }}" class="w-6 h-6" />
                                </div>
                                <div class="services-orbit-number">{{ str_pad($index + 1, 2, '0', STR_PAD_LEFT) }}</div>
                            </div>

                            {{-- Content --}}
                            <div class="services-orbit-content">
                                <h3 class="services-orbit-card-title">{{ $item['title'] ?? '' }}</h3>
                                <p class="services-orbit-desc">{{ $item['description'] ?? '' }}</p>

                                @if (!empty($item['price']))
                                    <div class="services-orbit-price">
                                        <x-icon icon="lucide:tag" class="w-4 h-4" />
                                        {{ $item['price'] }}
                                    </div>
                                @endif

                                {{-- Features --}}
                                @if (!empty($item['features']))
                                    <div class="services-orbit-features">
                                        @foreach (array_slice($item['features'], 0, 2) as $feature)
                                            <span class="services-orbit-feature">
                                                <x-icon icon="lucide:check" class="w-3.5 h-3.5" />
                                                {{ is_array($feature) ? ($feature['text'] ?? '') : $feature }}
                                            </span>
                                        @endforeach
                                    </div>
                                @endif
                            </div>

                            {{-- Button --}}
                            @if (!empty($item['button_text']) && !empty($item['button_url']))
                                <a href="{{ $item['button_url'] }}" class="services-orbit-btn">
                                    {{ $item['button_text'] }}
                                    <x-icon icon="{{ $item['button_icon'] ?? 'lucide:arrow-right' }}" class="w-4 h-4" />
                                </a>
                            @endif
                        </div>
                    </div>
                @endforeach
            </div>
        </div>
    </div>
</section>

<style>
    /* ===== Services Orbit Section ===== */
    .services-orbit-section {
        position: relative;
        padding: 6rem 0;
        overflow: hidden;
    }

    .services-orbit-white {
        background: #ffffff;
    }

    .services-orbit-gray {
        background: radial-gradient(ellipse at center, #f8fafc 0%, #e2e8f0 100%);
    }

    .dark .services-orbit-white {
        background: var(--color-bg-secondary);
    }

    .dark .services-orbit-gray {
        background: radial-gradient(ellipse at center, var(--color-bg-secondary) 0%, var(--color-bg-primary) 100%);
    }

    /* ===== Background ===== */
    .services-orbit-bg {
        position: absolute;
        inset: 0;
        pointer-events: none;
        overflow: hidden;
    }

    .services-orbit-circle {
        position: absolute;
        border-radius: 50%;
        border: 1px solid var(--color-primary-200);
        opacity: 0.3;
    }

    .services-orbit-circle-1 {
        width: 800px;
        height: 800px;
        top: 50%;
        left: 50%;
        transform: translate(-50%, -50%);
        animation: services-orbit-rotate 60s linear infinite;
    }

    .services-orbit-circle-2 {
        width: 600px;
        height: 600px;
        top: 50%;
        left: 50%;
        transform: translate(-50%, -50%);
        border-style: dashed;
        animation: services-orbit-rotate 45s linear infinite reverse;
    }

    .services-orbit-circle-3 {
        width: 400px;
        height: 400px;
        top: 50%;
        left: 50%;
        transform: translate(-50%, -50%);
        animation: services-orbit-rotate 30s linear infinite;
    }

    .dark .services-orbit-circle {
        border-color: var(--color-primary-800);
        opacity: 0.2;
    }

    @keyframes services-orbit-rotate {
        from { transform: translate(-50%, -50%) rotate(0deg); }
        to { transform: translate(-50%, -50%) rotate(360deg); }
    }

    .services-orbit-particles {
        position: absolute;
        inset: 0;
    }

    .services-orbit-particle {
        position: absolute;
        width: 4px;
        height: 4px;
        background: var(--color-primary-400);
        border-radius: 50%;
        left: var(--x);
        top: var(--y);
        opacity: 0.3;
        animation: services-orbit-twinkle 3s ease-in-out infinite;
        animation-delay: var(--delay);
    }

    @keyframes services-orbit-twinkle {
        0%, 100% { opacity: 0.2; transform: scale(1); }
        50% { opacity: 0.6; transform: scale(1.5); }
    }

    /* ===== Header ===== */
    .services-orbit-header {
        text-align: center;
        margin-bottom: 4rem;
        position: relative;
        z-index: 10;
    }

    .services-orbit-badge {
        display: inline-flex;
        align-items: center;
        gap: 0.75rem;
        padding: 0.5rem 1.25rem;
        background: white;
        border: 1px solid var(--color-gray-200);
        color: var(--color-primary-600);
        font-size: 0.8125rem;
        font-weight: 600;
        border-radius: 9999px;
        margin-bottom: 1.5rem;
        box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.05);
    }

    .dark .services-orbit-badge {
        background: var(--color-bg-tertiary);
        border-color: var(--color-border);
        color: var(--color-primary-400);
    }

    .services-orbit-badge-orbit {
        position: relative;
        width: 20px;
        height: 20px;
        border: 2px solid var(--color-primary-200);
        border-radius: 50%;
    }

    .services-orbit-badge-orbit span {
        position: absolute;
        width: 6px;
        height: 6px;
        background: var(--color-primary-500);
        border-radius: 50%;
        top: -3px;
        left: 50%;
        transform: translateX(-50%);
        animation: services-orbit-badge-spin 3s linear infinite;
        transform-origin: 50% calc(50% + 10px);
    }

    @keyframes services-orbit-badge-spin {
        from { transform: translateX(-50%) rotate(0deg); }
        to { transform: translateX(-50%) rotate(360deg); }
    }

    .services-orbit-title {
        font-family: var(--font-heading);
        font-size: 2.5rem;
        font-weight: 800;
        color: var(--color-gray-900);
        margin-bottom: 1rem;
    }

    .dark .services-orbit-title {
        color: var(--color-text-primary);
    }

    .services-orbit-subtitle {
        font-size: 1.125rem;
        color: var(--color-gray-600);
        max-width: 600px;
        margin: 0 auto;
    }

    .dark .services-orbit-subtitle {
        color: var(--color-text-tertiary);
    }

    /* ===== Container ===== */
    .services-orbit-container {
        position: relative;
    }

    /* ===== Central Hub ===== */
    .services-orbit-hub {
        display: none;
        position: absolute;
        top: 50%;
        left: 50%;
        transform: translate(-50%, -50%);
        z-index: 1;
    }

    @media (min-width: 1024px) {
        .services-orbit-hub {
            display: flex;
        }
    }

    .services-orbit-hub-inner {
        width: 160px;
        height: 160px;
        background: linear-gradient(135deg, var(--color-primary-500), var(--color-secondary-500));
        border-radius: 50%;
        display: flex;
        flex-direction: column;
        align-items: center;
        justify-content: center;
        color: white;
        position: relative;
        z-index: 2;
        box-shadow: 0 20px 60px -15px rgba(var(--color-primary-rgb, 37, 99, 235), 0.4);
    }

    .services-orbit-hub-icon {
        margin-bottom: 0.5rem;
    }

    .services-orbit-hub-text {
        text-align: center;
    }

    .services-orbit-hub-count {
        display: block;
        font-family: var(--font-heading);
        font-size: 2rem;
        font-weight: 800;
        line-height: 1;
    }

    .services-orbit-hub-label {
        font-size: 0.75rem;
        font-weight: 600;
        text-transform: uppercase;
        letter-spacing: 0.1em;
        opacity: 0.8;
    }

    .services-orbit-hub-ring {
        position: absolute;
        border-radius: 50%;
        border: 2px solid var(--color-primary-300);
    }

    .services-orbit-hub-ring-1 {
        inset: -20px;
        opacity: 0.5;
        animation: services-orbit-pulse 3s ease-out infinite;
    }

    .services-orbit-hub-ring-2 {
        inset: -40px;
        opacity: 0.3;
        animation: services-orbit-pulse 3s ease-out infinite 1s;
    }

    @keyframes services-orbit-pulse {
        0% { transform: scale(1); opacity: 0.5; }
        100% { transform: scale(1.3); opacity: 0; }
    }

    /* ===== Cards Grid ===== */
    .services-orbit-cards {
        display: grid;
        gap: 1.5rem;
    }

    @media (min-width: 640px) {
        .services-orbit-cards {
            grid-template-columns: repeat(2, 1fr);
        }
    }

    @media (min-width: 1024px) {
        .services-orbit-cards {
            grid-template-columns: repeat(3, 1fr);
            gap: 2rem;
        }
    }

    /* ===== Card ===== */
    .services-orbit-card {
        position: relative;
        z-index: 10;
    }

    .services-orbit-card-connector {
        display: none;
    }

    .services-orbit-card-inner {
        background: white;
        border: 1px solid var(--color-gray-200);
        border-radius: 1.25rem;
        padding: 1.5rem;
        height: 100%;
        display: flex;
        flex-direction: column;
        transition: all 0.4s cubic-bezier(0.4, 0, 0.2, 1);
        position: relative;
        overflow: hidden;
    }

    .services-orbit-card-inner::before {
        content: '';
        position: absolute;
        top: 0;
        left: 0;
        right: 0;
        height: 4px;
        background: linear-gradient(90deg, var(--color-primary-500), var(--color-secondary-500));
        transform: scaleX(0);
        transform-origin: left;
        transition: transform 0.4s ease;
    }

    .services-orbit-card:hover .services-orbit-card-inner::before {
        transform: scaleX(1);
    }

    .services-orbit-card:hover .services-orbit-card-inner {
        transform: translateY(-8px);
        border-color: var(--color-primary-200);
        box-shadow: 0 20px 50px -15px rgba(0, 0, 0, 0.15);
    }

    .dark .services-orbit-card-inner {
        background: var(--color-card-bg);
        border-color: var(--color-border);
    }

    .dark .services-orbit-card:hover .services-orbit-card-inner {
        border-color: var(--color-primary-700);
        box-shadow: 0 20px 50px -15px rgba(0, 0, 0, 0.4);
    }

    /* ===== Card Header ===== */
    .services-orbit-card-header {
        display: flex;
        align-items: center;
        justify-content: space-between;
        margin-bottom: 1rem;
    }

    .services-orbit-icon {
        width: 3rem;
        height: 3rem;
        display: flex;
        align-items: center;
        justify-content: center;
        background: linear-gradient(135deg, var(--color-primary-500), var(--color-secondary-500));
        border-radius: 0.875rem;
        color: white;
        transition: all 0.4s ease;
    }

    .services-orbit-card:hover .services-orbit-icon {
        transform: rotate(-10deg) scale(1.1);
    }

    .services-orbit-number {
        font-family: var(--font-heading);
        font-size: 2rem;
        font-weight: 800;
        color: var(--color-gray-100);
        line-height: 1;
    }

    .dark .services-orbit-number {
        color: var(--color-gray-800);
    }

    /* ===== Content ===== */
    .services-orbit-content {
        flex: 1;
    }

    .services-orbit-card-title {
        font-family: var(--font-heading);
        font-size: 1.25rem;
        font-weight: 700;
        color: var(--color-gray-900);
        margin-bottom: 0.5rem;
    }

    .dark .services-orbit-card-title {
        color: var(--color-text-primary);
    }

    .services-orbit-desc {
        font-size: 0.9375rem;
        color: var(--color-gray-600);
        line-height: 1.6;
        margin-bottom: 1rem;
    }

    .dark .services-orbit-desc {
        color: var(--color-text-tertiary);
    }

    .services-orbit-price {
        display: inline-flex;
        align-items: center;
        gap: 0.375rem;
        font-size: 0.875rem;
        font-weight: 700;
        color: var(--color-primary-600);
        padding: 0.375rem 0.875rem;
        background: var(--color-primary-50);
        border-radius: 9999px;
        margin-bottom: 1rem;
    }

    .dark .services-orbit-price {
        color: var(--color-primary-400);
        background: rgba(var(--color-primary-rgb, 37, 99, 235), 0.15);
    }

    /* ===== Features ===== */
    .services-orbit-features {
        display: flex;
        flex-direction: column;
        gap: 0.375rem;
        margin-bottom: 1rem;
    }

    .services-orbit-feature {
        display: flex;
        align-items: center;
        gap: 0.375rem;
        font-size: 0.8125rem;
        color: var(--color-gray-600);
    }

    .services-orbit-feature svg {
        color: var(--color-primary-500);
        flex-shrink: 0;
    }

    .dark .services-orbit-feature {
        color: var(--color-text-tertiary);
    }

    .dark .services-orbit-feature svg {
        color: var(--color-primary-400);
    }

    /* ===== Button ===== */
    .services-orbit-btn {
        display: inline-flex;
        align-items: center;
        gap: 0.5rem;
        font-size: 0.875rem;
        font-weight: 600;
        color: var(--color-primary-600);
        padding: 0.625rem 1rem;
        border: 1px solid var(--color-primary-200);
        border-radius: 0.75rem;
        transition: all 0.3s ease;
        margin-top: auto;
    }

    .services-orbit-btn:hover {
        background: var(--color-primary-600);
        border-color: var(--color-primary-600);
        color: white;
    }

    .dark .services-orbit-btn {
        color: var(--color-primary-400);
        border-color: var(--color-primary-700);
    }

    .dark .services-orbit-btn:hover {
        background: var(--color-primary-600);
        border-color: var(--color-primary-600);
        color: white;
    }

    /* ===== RTL Support ===== */
    [dir="rtl"] .services-orbit-section {
        direction: rtl;
    }

    [dir="rtl"] .services-orbit-badge {
        flex-direction: row-reverse;
    }

    [dir="rtl"] .services-orbit-card-header {
        flex-direction: row-reverse;
    }

    [dir="rtl"] .services-orbit-content {
        text-align: right;
    }

    [dir="rtl"] .services-orbit-price {
        flex-direction: row-reverse;
    }

    [dir="rtl"] .services-orbit-feature {
        flex-direction: row-reverse;
    }

    [dir="rtl"] .services-orbit-btn {
        flex-direction: row-reverse;
    }

    [dir="rtl"] .services-orbit-btn svg {
        transform: scaleX(-1);
    }

    [dir="rtl"] .services-orbit-card:hover .services-orbit-icon {
        transform: rotate(10deg) scale(1.1);
    }

    [dir="rtl"] .services-orbit-card-inner::before {
        transform-origin: right;
    }

    /* ===== Responsive ===== */
    @media (max-width: 640px) {
        .services-orbit-section {
            padding: 4rem 0;
        }

        .services-orbit-title {
            font-size: 2rem;
        }

        .services-orbit-card-inner {
            padding: 1.25rem;
        }

        .services-orbit-btn {
            width: 100%;
            justify-content: center;
        }

        .services-orbit-circle {
            display: none;
        }
    }
</style>
</div>
