<?php

use Livewire\Component;

new class extends Component {
    public array $section = [];
    public array $items = [];
    public array $company = [];
    public array $social = [];
    public array $services = [];
    public bool $isWhiteBackground = false;
    public ?int $paddingTop = null;
    public ?int $paddingBottom = null;

    private array $defaultItems = [
        [
            'icon' => 'lucide:shirt',
            'title' => 'Wash & Fold',
            'description' => 'Professional washing, drying, and folding service for your everyday clothes.',
            'price' => 'From $2.50/lb',
            'features' => [
                ['text' => 'Same-day service available'],
                ['text' => 'Eco-friendly detergents'],
                ['text' => 'Fabric softener included'],
            ],
            'button_text' => 'Learn More',
            'button_url' => '#contact',
            'button_icon' => 'lucide:arrow-right',
        ],
        [
            'icon' => 'lucide:sparkles',
            'title' => 'Dry Cleaning',
            'description' => 'Expert care for your delicate and special garments that need extra attention.',
            'price' => 'From $8.99/item',
            'features' => [
                ['text' => 'Suits & formal wear'],
                ['text' => 'Delicate fabrics'],
                ['text' => 'Stain treatment included'],
            ],
            'button_text' => 'Learn More',
            'button_url' => '#contact',
            'button_icon' => 'lucide:arrow-right',
        ],
        [
            'icon' => 'lucide:wind',
            'title' => 'Press & Iron',
            'description' => 'Professional pressing and ironing for a crisp, polished look.',
            'price' => 'From $3.50/item',
            'features' => [
                ['text' => 'Steam pressing'],
                ['text' => 'Hand finishing'],
                ['text' => 'Ready to wear'],
            ],
            'button_text' => 'Learn More',
            'button_url' => '#contact',
            'button_icon' => 'lucide:arrow-right',
        ],
        [
            'icon' => 'lucide:bed-double',
            'title' => 'Bedding & Linens',
            'description' => 'Fresh, clean bedding and household linens for maximum comfort.',
            'price' => 'From $15.99/set',
            'features' => [
                ['text' => 'Comforters & duvets'],
                ['text' => 'Sheets & pillowcases'],
                ['text' => 'Allergen removal'],
            ],
            'button_text' => 'Learn More',
            'button_url' => '#contact',
            'button_icon' => 'lucide:arrow-right',
        ],
        [
            'icon' => 'lucide:gem',
            'title' => 'Premium Care',
            'description' => 'Specialized treatment for luxury fabrics and designer pieces.',
            'price' => 'Custom pricing',
            'features' => [
                ['text' => 'Designer labels'],
                ['text' => 'Silk & cashmere'],
                ['text' => 'Hand wash only items'],
            ],
            'button_text' => 'Learn More',
            'button_url' => '#contact',
            'button_icon' => 'lucide:arrow-right',
        ],
    ];

    public function mount(
        array $section = [],
        array $items = [],
        array $company = [],
        array $social = [],
        array $services = [],
        bool $isWhiteBackground = false,
        ?int $paddingTop = null,
        ?int $paddingBottom = null
    ) {
        $this->company = $company;
        $this->social = $social;
        $this->isWhiteBackground = $isWhiteBackground;
        $this->paddingTop = $paddingTop;
        $this->paddingBottom = $paddingBottom;

        $this->section = array_merge([
            'badge' => 'Our Services',
            'title' => 'Professional Laundry Services',
            'subtitle' => 'From everyday laundry to delicate dry cleaning, we\'ve got all your garment care needs covered.',
        ], $section);

        $this->items = !empty($section['items']) ? $section['items'] : (!empty($items) ? $items : $this->defaultItems);
    }

    public function getCustomPaddingStyle(): string
    {
        $styles = [];
        if ($this->paddingTop !== null) {
            $styles[] = "padding-top: {$this->paddingTop}px";
        }
        if ($this->paddingBottom !== null) {
            $styles[] = "padding-bottom: {$this->paddingBottom}px";
        }
        return !empty($styles) ? implode('; ', $styles) : '';
    }
};
?>

<div>
<section id="services" class="services-timeline-section {{ $isWhiteBackground ? 'services-timeline-white' : 'services-timeline-gray' }}" @if($this->getCustomPaddingStyle()) style="{{ $this->getCustomPaddingStyle() }}" @endif>
    {{-- Background Elements --}}
    <div class="services-timeline-bg">
        <div class="services-timeline-shape services-timeline-shape-1"></div>
        <div class="services-timeline-shape services-timeline-shape-2"></div>
        <div class="services-timeline-shape services-timeline-shape-3"></div>
    </div>

    <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 relative">
        {{-- Section Header --}}
        <div class="services-timeline-header">
            <div class="services-timeline-badge">
                <span class="services-timeline-badge-dot"></span>
                {{ $section['badge'] }}
            </div>
            <h2 class="services-timeline-title">{{ $section['title'] }}</h2>
            <p class="services-timeline-subtitle">{{ $section['subtitle'] }}</p>
        </div>

        {{-- Timeline Container --}}
        <div class="services-timeline-container">
            {{-- Central Line --}}
            <div class="services-timeline-line">
                <div class="services-timeline-line-progress"></div>
            </div>

            {{-- Timeline Items --}}
            @foreach ($items as $index => $item)
                <div class="services-timeline-item scroll-animate {{ $index % 2 === 0 ? 'services-timeline-item-left' : 'services-timeline-item-right' }}" style="animation-delay: {{ $index * 150 }}ms">
                    {{-- Timeline Node --}}
                    <div class="services-timeline-node">
                        <div class="services-timeline-node-outer">
                            <div class="services-timeline-node-inner">
                                <span>{{ str_pad($index + 1, 2, '0', STR_PAD_LEFT) }}</span>
                            </div>
                        </div>
                        <div class="services-timeline-node-pulse"></div>
                    </div>

                    {{-- Card --}}
                    <div class="services-timeline-card">
                        <div class="services-timeline-card-arrow"></div>
                        <div class="services-timeline-card-inner">
                            {{-- Icon --}}
                            <div class="services-timeline-icon">
                                <x-icon icon="{{ $item['icon'] ?? 'heroicons:sparkles' }}" class="w-6 h-6" />
                            </div>

                            {{-- Content --}}
                            <div class="services-timeline-content">
                                <div class="services-timeline-content-header">
                                    <h3 class="services-timeline-card-title">{{ $item['title'] ?? '' }}</h3>
                                    @if (!empty($item['price']))
                                        <div class="services-timeline-price">{{ $item['price'] }}</div>
                                    @endif
                                </div>
                                <p class="services-timeline-desc">{{ $item['description'] ?? '' }}</p>

                                {{-- Features --}}
                                @if (!empty($item['features']))
                                    <div class="services-timeline-features">
                                        @foreach ($item['features'] as $feature)
                                            <span class="services-timeline-feature">
                                                <x-icon icon="lucide:check" class="w-3.5 h-3.5" />
                                                {{ is_array($feature) ? ($feature['text'] ?? '') : $feature }}
                                            </span>
                                        @endforeach
                                    </div>
                                @endif

                                {{-- Button --}}
                                @if (!empty($item['button_text']) && !empty($item['button_url']))
                                    <a href="{{ $item['button_url'] }}" class="services-timeline-btn">
                                        {{ $item['button_text'] }}
                                        <x-icon icon="{{ $item['button_icon'] ?? 'lucide:chevron-right' }}" class="w-4 h-4" />
                                    </a>
                                @endif
                            </div>
                        </div>
                    </div>
                </div>
            @endforeach
        </div>
    </div>
</section>

<style>
    /* ===== Services Timeline Section ===== */
    .services-timeline-section {
        position: relative;
        padding: 6rem 0;
        overflow: hidden;
    }

    .services-timeline-white {
        background: #ffffff;
    }

    .services-timeline-gray {
        background: linear-gradient(135deg, #f8fafc 0%, #f1f5f9 100%);
    }

    .dark .services-timeline-white {
        background: var(--color-bg-secondary);
    }

    .dark .services-timeline-gray {
        background: linear-gradient(135deg, var(--color-bg-primary) 0%, var(--color-bg-secondary) 100%);
    }

    /* ===== Background ===== */
    .services-timeline-bg {
        position: absolute;
        inset: 0;
        pointer-events: none;
        overflow: hidden;
    }

    .services-timeline-shape {
        position: absolute;
        border-radius: 50%;
        background: linear-gradient(135deg, var(--color-primary-200), var(--color-secondary-200));
        opacity: 0.3;
    }

    .services-timeline-shape-1 {
        width: 400px;
        height: 400px;
        top: -200px;
        left: -100px;
    }

    .services-timeline-shape-2 {
        width: 300px;
        height: 300px;
        top: 40%;
        right: -150px;
    }

    .services-timeline-shape-3 {
        width: 250px;
        height: 250px;
        bottom: -100px;
        left: 20%;
    }

    .dark .services-timeline-shape {
        background: linear-gradient(135deg, var(--color-primary-900), var(--color-secondary-900));
        opacity: 0.2;
    }

    /* ===== Header ===== */
    .services-timeline-header {
        text-align: center;
        margin-bottom: 5rem;
    }

    .services-timeline-badge {
        display: inline-flex;
        align-items: center;
        gap: 0.5rem;
        padding: 0.5rem 1.25rem;
        background: white;
        border: 1px solid var(--color-gray-200);
        color: var(--color-primary-600);
        font-size: 0.8125rem;
        font-weight: 600;
        border-radius: 9999px;
        margin-bottom: 1.5rem;
        box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.05);
    }

    .dark .services-timeline-badge {
        background: var(--color-bg-tertiary);
        border-color: var(--color-border);
        color: var(--color-primary-400);
    }

    .services-timeline-badge-dot {
        width: 8px;
        height: 8px;
        background: var(--color-primary-500);
        border-radius: 50%;
        animation: services-timeline-pulse 2s ease-in-out infinite;
    }

    @keyframes services-timeline-pulse {
        0%, 100% { transform: scale(1); opacity: 1; }
        50% { transform: scale(1.3); opacity: 0.7; }
    }

    .services-timeline-title {
        font-family: var(--font-heading);
        font-size: 2.5rem;
        font-weight: 800;
        color: var(--color-gray-900);
        margin-bottom: 1rem;
    }

    .dark .services-timeline-title {
        color: var(--color-text-primary);
    }

    .services-timeline-subtitle {
        font-size: 1.125rem;
        color: var(--color-gray-600);
        max-width: 600px;
        margin: 0 auto;
    }

    .dark .services-timeline-subtitle {
        color: var(--color-text-tertiary);
    }

    /* ===== Timeline Container ===== */
    .services-timeline-container {
        position: relative;
        max-width: 1000px;
        margin: 0 auto;
    }

    /* ===== Timeline Line ===== */
    .services-timeline-line {
        position: absolute;
        left: 50%;
        top: 0;
        bottom: 0;
        width: 4px;
        background: var(--color-gray-200);
        transform: translateX(-50%);
        border-radius: 2px;
    }

    .dark .services-timeline-line {
        background: var(--color-border);
    }

    .services-timeline-line-progress {
        width: 100%;
        height: 100%;
        background: linear-gradient(180deg, var(--color-primary-500), var(--color-secondary-500));
        border-radius: 2px;
        transform-origin: top;
        animation: services-timeline-fill 2s ease-out forwards;
    }

    @keyframes services-timeline-fill {
        from { transform: scaleY(0); }
        to { transform: scaleY(1); }
    }

    /* ===== Timeline Item ===== */
    .services-timeline-item {
        position: relative;
        display: flex;
        align-items: flex-start;
        margin-bottom: 3rem;
    }

    .services-timeline-item:last-child {
        margin-bottom: 0;
    }

    .services-timeline-item-left {
        flex-direction: row;
        padding-right: calc(50% + 2rem);
    }

    .services-timeline-item-right {
        flex-direction: row-reverse;
        padding-left: calc(50% + 2rem);
    }

    /* ===== Timeline Node ===== */
    .services-timeline-node {
        position: absolute;
        left: 50%;
        transform: translateX(-50%);
        z-index: 2;
    }

    .services-timeline-node-outer {
        width: 60px;
        height: 60px;
        background: white;
        border: 3px solid var(--color-primary-500);
        border-radius: 50%;
        display: flex;
        align-items: center;
        justify-content: center;
        box-shadow: 0 4px 15px rgba(var(--color-primary-rgb, 37, 99, 235), 0.2);
        transition: all 0.3s ease;
    }

    .services-timeline-item:hover .services-timeline-node-outer {
        transform: scale(1.1);
        border-color: var(--color-secondary-500);
    }

    .dark .services-timeline-node-outer {
        background: var(--color-bg-secondary);
    }

    .services-timeline-node-inner {
        width: 40px;
        height: 40px;
        background: linear-gradient(135deg, var(--color-primary-500), var(--color-secondary-500));
        border-radius: 50%;
        display: flex;
        align-items: center;
        justify-content: center;
        color: white;
        font-family: var(--font-heading);
        font-size: 0.875rem;
        font-weight: 700;
    }

    .services-timeline-node-pulse {
        position: absolute;
        inset: -10px;
        border: 2px solid var(--color-primary-300);
        border-radius: 50%;
        opacity: 0;
        animation: services-timeline-ring 2s ease-out infinite;
    }

    @keyframes services-timeline-ring {
        0% { transform: scale(0.8); opacity: 0.8; }
        100% { transform: scale(1.5); opacity: 0; }
    }

    /* ===== Timeline Card ===== */
    .services-timeline-card {
        position: relative;
        flex: 1;
    }

    .services-timeline-card-arrow {
        position: absolute;
        top: 20px;
        width: 16px;
        height: 16px;
        background: white;
        transform: rotate(45deg);
        border: 1px solid var(--color-gray-200);
    }

    .services-timeline-item-left .services-timeline-card-arrow {
        right: -8px;
        border-left: none;
        border-bottom: none;
    }

    .services-timeline-item-right .services-timeline-card-arrow {
        left: -8px;
        border-right: none;
        border-top: none;
    }

    .dark .services-timeline-card-arrow {
        background: var(--color-card-bg);
        border-color: var(--color-border);
    }

    .services-timeline-card-inner {
        background: white;
        border: 1px solid var(--color-gray-200);
        border-radius: 1.25rem;
        padding: 1.5rem;
        box-shadow: 0 10px 40px -10px rgba(0, 0, 0, 0.1);
        transition: all 0.4s ease;
    }

    .services-timeline-item:hover .services-timeline-card-inner {
        transform: translateY(-5px);
        box-shadow: 0 20px 50px -15px rgba(0, 0, 0, 0.15);
    }

    .dark .services-timeline-card-inner {
        background: var(--color-card-bg);
        border-color: var(--color-border);
        box-shadow: 0 10px 40px -10px rgba(0, 0, 0, 0.3);
    }

    .dark .services-timeline-item:hover .services-timeline-card-inner {
        box-shadow: 0 20px 50px -15px rgba(0, 0, 0, 0.5);
    }

    /* ===== Icon ===== */
    .services-timeline-icon {
        width: 3rem;
        height: 3rem;
        display: flex;
        align-items: center;
        justify-content: center;
        background: linear-gradient(135deg, var(--color-primary-500), var(--color-secondary-500));
        border-radius: 0.75rem;
        color: white;
        margin-bottom: 1rem;
        transition: all 0.3s ease;
    }

    .services-timeline-item:hover .services-timeline-icon {
        transform: rotate(-10deg) scale(1.1);
    }

    /* ===== Content ===== */
    .services-timeline-content-header {
        display: flex;
        align-items: center;
        justify-content: space-between;
        gap: 1rem;
        margin-bottom: 0.5rem;
        flex-wrap: wrap;
    }

    .services-timeline-card-title {
        font-family: var(--font-heading);
        font-size: 1.25rem;
        font-weight: 700;
        color: var(--color-gray-900);
    }

    .dark .services-timeline-card-title {
        color: var(--color-text-primary);
    }

    .services-timeline-price {
        font-size: 0.8125rem;
        font-weight: 700;
        color: var(--color-primary-600);
        padding: 0.25rem 0.75rem;
        background: var(--color-primary-50);
        border-radius: 9999px;
    }

    .dark .services-timeline-price {
        color: var(--color-primary-400);
        background: rgba(var(--color-primary-rgb, 37, 99, 235), 0.15);
    }

    .services-timeline-desc {
        font-size: 0.9375rem;
        color: var(--color-gray-600);
        line-height: 1.6;
        margin-bottom: 1rem;
    }

    .dark .services-timeline-desc {
        color: var(--color-text-tertiary);
    }

    /* ===== Features ===== */
    .services-timeline-features {
        display: flex;
        flex-wrap: wrap;
        gap: 0.5rem;
        margin-bottom: 1rem;
    }

    .services-timeline-feature {
        display: inline-flex;
        align-items: center;
        gap: 0.25rem;
        font-size: 0.8125rem;
        color: var(--color-gray-600);
        padding: 0.25rem 0.625rem;
        background: var(--color-gray-100);
        border-radius: 9999px;
    }

    .services-timeline-feature svg {
        color: var(--color-primary-500);
    }

    .dark .services-timeline-feature {
        color: var(--color-text-tertiary);
        background: var(--color-bg-tertiary);
    }

    .dark .services-timeline-feature svg {
        color: var(--color-primary-400);
    }

    /* ===== Button ===== */
    .services-timeline-btn {
        display: inline-flex;
        align-items: center;
        gap: 0.375rem;
        font-size: 0.875rem;
        font-weight: 600;
        color: var(--color-primary-600);
        transition: all 0.3s ease;
    }

    .services-timeline-btn:hover {
        color: var(--color-primary-700);
        gap: 0.625rem;
    }

    .dark .services-timeline-btn {
        color: var(--color-primary-400);
    }

    .dark .services-timeline-btn:hover {
        color: var(--color-primary-300);
    }

    /* ===== RTL Support ===== */
    [dir="rtl"] .services-timeline-section {
        direction: rtl;
    }

    [dir="rtl"] .services-timeline-badge {
        flex-direction: row-reverse;
    }

    [dir="rtl"] .services-timeline-item-left {
        flex-direction: row-reverse;
        padding-right: 0;
        padding-left: calc(50% + 2rem);
    }

    [dir="rtl"] .services-timeline-item-right {
        flex-direction: row;
        padding-left: 0;
        padding-right: calc(50% + 2rem);
    }

    [dir="rtl"] .services-timeline-item-left .services-timeline-card-arrow {
        right: auto;
        left: -8px;
        border-left: 1px solid var(--color-gray-200);
        border-bottom: 1px solid var(--color-gray-200);
        border-right: none;
        border-top: none;
    }

    [dir="rtl"] .services-timeline-item-right .services-timeline-card-arrow {
        left: auto;
        right: -8px;
        border-right: 1px solid var(--color-gray-200);
        border-top: 1px solid var(--color-gray-200);
        border-left: none;
        border-bottom: none;
    }

    [dir="rtl"] .services-timeline-content {
        text-align: right;
    }

    [dir="rtl"] .services-timeline-content-header {
        flex-direction: row-reverse;
    }

    [dir="rtl"] .services-timeline-features {
        justify-content: flex-end;
    }

    [dir="rtl"] .services-timeline-feature {
        flex-direction: row-reverse;
    }

    [dir="rtl"] .services-timeline-btn {
        flex-direction: row-reverse;
    }

    [dir="rtl"] .services-timeline-btn svg {
        transform: scaleX(-1);
    }

    [dir="rtl"] .services-timeline-item:hover .services-timeline-icon {
        transform: rotate(10deg) scale(1.1);
    }

    /* ===== Responsive ===== */
    @media (max-width: 768px) {
        .services-timeline-section {
            padding: 4rem 0;
        }

        .services-timeline-title {
            font-size: 2rem;
        }

        .services-timeline-line {
            left: 30px;
        }

        .services-timeline-node {
            left: 30px;
        }

        .services-timeline-node-outer {
            width: 50px;
            height: 50px;
        }

        .services-timeline-node-inner {
            width: 32px;
            height: 32px;
            font-size: 0.75rem;
        }

        .services-timeline-item-left,
        .services-timeline-item-right {
            flex-direction: row;
            padding-left: 80px;
            padding-right: 0;
        }

        [dir="rtl"] .services-timeline-line {
            left: auto;
            right: 30px;
        }

        [dir="rtl"] .services-timeline-node {
            left: auto;
            right: 30px;
            transform: translateX(50%);
        }

        [dir="rtl"] .services-timeline-item-left,
        [dir="rtl"] .services-timeline-item-right {
            flex-direction: row-reverse;
            padding-left: 0;
            padding-right: 80px;
        }

        .services-timeline-card-arrow {
            display: none;
        }

        .services-timeline-content-header {
            flex-direction: column;
            align-items: flex-start;
        }

        [dir="rtl"] .services-timeline-content-header {
            align-items: flex-end;
        }
    }
</style>
</div>
