<?php

use Livewire\Component;

new class extends Component {
    public array $section = [];
    public array $items = [];
    public array $company = [];
    public array $social = [];
    public array $services = [];
    public bool $isWhiteBackground = false;
    public ?int $paddingTop = null;
    public ?int $paddingBottom = null;

    // Default static data for services
    private array $defaultItems = [
        [
            'icon' => 'lucide:shirt',
            'title' => 'Wash & Fold',
            'description' => 'Professional washing, drying, and folding service for your everyday clothes.',
            'price' => 'From $2.50/lb',
            'features' => [
                ['text' => 'Same-day service available'],
                ['text' => 'Eco-friendly detergents'],
                ['text' => 'Fabric softener included'],
            ],
            'button_text' => 'Learn More',
            'button_url' => '#contact',
            'button_icon' => 'lucide:arrow-right',
        ],
        [
            'icon' => 'lucide:sparkles',
            'title' => 'Dry Cleaning',
            'description' => 'Expert care for your delicate and special garments that need extra attention.',
            'price' => 'From $8.99/item',
            'features' => [
                ['text' => 'Suits & formal wear'],
                ['text' => 'Delicate fabrics'],
                ['text' => 'Stain treatment included'],
            ],
            'button_text' => 'Learn More',
            'button_url' => '#contact',
            'button_icon' => 'lucide:arrow-right',
        ],
        [
            'icon' => 'lucide:wind',
            'title' => 'Press & Iron',
            'description' => 'Professional pressing and ironing for a crisp, polished look.',
            'price' => 'From $3.50/item',
            'features' => [
                ['text' => 'Steam pressing'],
                ['text' => 'Hand finishing'],
                ['text' => 'Ready to wear'],
            ],
            'button_text' => 'Learn More',
            'button_url' => '#contact',
            'button_icon' => 'lucide:arrow-right',
        ],
        [
            'icon' => 'lucide:bed-double',
            'title' => 'Bedding & Linens',
            'description' => 'Fresh, clean bedding and household linens for maximum comfort.',
            'price' => 'From $15.99/set',
            'features' => [
                ['text' => 'Comforters & duvets'],
                ['text' => 'Sheets & pillowcases'],
                ['text' => 'Allergen removal'],
            ],
            'button_text' => 'Learn More',
            'button_url' => '#contact',
            'button_icon' => 'lucide:arrow-right',
        ],
    ];

    public function mount(
        array $section = [],
        array $items = [],
        array $company = [],
        array $social = [],
        array $services = [],
        bool $isWhiteBackground = false,
        ?int $paddingTop = null,
        ?int $paddingBottom = null
    ) {
        $this->company = $company;
        $this->social = $social;
        $this->isWhiteBackground = $isWhiteBackground;
        $this->paddingTop = $paddingTop;
        $this->paddingBottom = $paddingBottom;

        $this->section = array_merge([
            'badge' => 'Our Services',
            'title' => 'Professional Laundry Services',
            'subtitle' => 'From everyday laundry to delicate dry cleaning, we\'ve got all your garment care needs covered.',
        ], $section);

        $this->items = !empty($section['items']) ? $section['items'] : (!empty($items) ? $items : $this->defaultItems);
    }

    public function getCustomPaddingStyle(): string
    {
        $styles = [];
        if ($this->paddingTop !== null) {
            $styles[] = "padding-top: {$this->paddingTop}px";
        }
        if ($this->paddingBottom !== null) {
            $styles[] = "padding-bottom: {$this->paddingBottom}px";
        }
        return !empty($styles) ? implode('; ', $styles) : '';
    }
};
?>

<div>
<section id="services" class="services-zigzag-section {{ $isWhiteBackground ? 'services-zigzag-white' : 'services-zigzag-gray' }}" @if($this->getCustomPaddingStyle()) style="{{ $this->getCustomPaddingStyle() }}" @endif>
    {{-- Background Elements --}}
    <div class="services-zigzag-bg">
        <div class="services-zigzag-wave services-zigzag-wave-1"></div>
        <div class="services-zigzag-wave services-zigzag-wave-2"></div>
        <div class="services-zigzag-dot services-zigzag-dot-1"></div>
        <div class="services-zigzag-dot services-zigzag-dot-2"></div>
        <div class="services-zigzag-dot services-zigzag-dot-3"></div>
    </div>

    <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 relative">
        {{-- Section Header --}}
        <div class="services-zigzag-header">
            <div class="services-zigzag-badge">
                <span class="services-zigzag-badge-icon">
                    <x-icon icon="lucide:layers" class="w-4 h-4" />
                </span>
                {{ $section['badge'] }}
            </div>
            <h2 class="services-zigzag-title">{{ $section['title'] }}</h2>
            <p class="services-zigzag-subtitle">{{ $section['subtitle'] }}</p>
        </div>

        {{-- Zigzag Items --}}
        <div class="services-zigzag-list">
            @foreach ($items as $index => $item)
                <div class="services-zigzag-item scroll-animate {{ $index % 2 === 1 ? 'services-zigzag-item-reverse' : '' }}" style="animation-delay: {{ $index * 150 }}ms">
                    {{-- Content Side --}}
                    <div class="services-zigzag-content">
                        <div class="services-zigzag-number">{{ str_pad($index + 1, 2, '0', STR_PAD_LEFT) }}</div>
                        <h3 class="services-zigzag-item-title">{{ $item['title'] ?? '' }}</h3>
                        <p class="services-zigzag-desc">{{ $item['description'] ?? '' }}</p>

                        @if (!empty($item['features']))
                            <div class="services-zigzag-features">
                                @foreach ($item['features'] as $feature)
                                    <div class="services-zigzag-feature">
                                        <x-icon icon="lucide:check-circle-2" class="w-5 h-5" />
                                        <span>{{ is_array($feature) ? ($feature['text'] ?? '') : $feature }}</span>
                                    </div>
                                @endforeach
                            </div>
                        @endif

                        <div class="services-zigzag-footer">
                            @if (!empty($item['price']))
                                <div class="services-zigzag-price">{{ $item['price'] }}</div>
                            @endif
                            @if (!empty($item['button_text']) && !empty($item['button_url']))
                                <a href="{{ $item['button_url'] }}" class="services-zigzag-btn">
                                    <span>{{ $item['button_text'] }}</span>
                                    <x-icon icon="{{ $item['button_icon'] ?? 'lucide:arrow-right' }}" class="w-4 h-4" />
                                </a>
                            @endif
                        </div>
                    </div>

                    {{-- Icon Side --}}
                    <div class="services-zigzag-visual">
                        <div class="services-zigzag-icon-wrapper">
                            <div class="services-zigzag-icon-bg"></div>
                            <div class="services-zigzag-icon">
                                <x-icon icon="{{ $item['icon'] ?? 'heroicons:sparkles' }}" class="w-12 h-12" />
                            </div>
                            <div class="services-zigzag-icon-ring"></div>
                            <div class="services-zigzag-icon-dots">
                                <span></span><span></span><span></span>
                            </div>
                        </div>
                    </div>

                    {{-- Connector Line --}}
                    @if (!$loop->last)
                        <div class="services-zigzag-connector"></div>
                    @endif
                </div>
            @endforeach
        </div>
    </div>
</section>

<style>
    /* ===== Services Zigzag Section ===== */
    .services-zigzag-section {
        position: relative;
        padding: 6rem 0;
        overflow: hidden;
    }

    .services-zigzag-white {
        background: #ffffff;
    }

    .services-zigzag-gray {
        background: linear-gradient(180deg, #f8fafc 0%, #ffffff 100%);
    }

    .dark .services-zigzag-white {
        background: var(--color-bg-secondary);
    }

    .dark .services-zigzag-gray {
        background: linear-gradient(180deg, var(--color-bg-primary) 0%, var(--color-bg-secondary) 100%);
    }

    /* ===== Background ===== */
    .services-zigzag-bg {
        position: absolute;
        inset: 0;
        pointer-events: none;
        overflow: hidden;
    }

    .services-zigzag-wave {
        position: absolute;
        width: 100%;
        height: 200px;
        background: linear-gradient(90deg, transparent, var(--color-primary-100), transparent);
        opacity: 0.3;
    }

    .services-zigzag-wave-1 {
        top: 20%;
        transform: skewY(-3deg);
    }

    .services-zigzag-wave-2 {
        bottom: 30%;
        transform: skewY(3deg);
        background: linear-gradient(90deg, transparent, var(--color-secondary-100), transparent);
    }

    .dark .services-zigzag-wave {
        background: linear-gradient(90deg, transparent, var(--color-primary-900), transparent);
        opacity: 0.2;
    }

    .dark .services-zigzag-wave-2 {
        background: linear-gradient(90deg, transparent, var(--color-secondary-900), transparent);
    }

    .services-zigzag-dot {
        position: absolute;
        width: 8px;
        height: 8px;
        background: var(--color-primary-400);
        border-radius: 50%;
        opacity: 0.5;
    }

    .services-zigzag-dot-1 { top: 15%; left: 10%; }
    .services-zigzag-dot-2 { top: 45%; right: 8%; }
    .services-zigzag-dot-3 { bottom: 20%; left: 15%; }

    .dark .services-zigzag-dot {
        background: var(--color-primary-500);
        opacity: 0.3;
    }

    /* ===== Header ===== */
    .services-zigzag-header {
        text-align: center;
        margin-bottom: 5rem;
    }

    .services-zigzag-badge {
        display: inline-flex;
        align-items: center;
        gap: 0.5rem;
        padding: 0.5rem 1.25rem;
        background: linear-gradient(135deg, var(--color-primary-500), var(--color-secondary-500));
        color: white;
        font-size: 0.8125rem;
        font-weight: 600;
        border-radius: 9999px;
        margin-bottom: 1.5rem;
    }

    .services-zigzag-badge-icon {
        display: flex;
        align-items: center;
        justify-content: center;
    }

    .services-zigzag-title {
        font-family: var(--font-heading);
        font-size: 2.5rem;
        font-weight: 800;
        color: var(--color-gray-900);
        margin-bottom: 1rem;
    }

    .dark .services-zigzag-title {
        color: var(--color-text-primary);
    }

    .services-zigzag-subtitle {
        font-size: 1.125rem;
        color: var(--color-gray-600);
        max-width: 600px;
        margin: 0 auto;
    }

    .dark .services-zigzag-subtitle {
        color: var(--color-text-tertiary);
    }

    /* ===== List ===== */
    .services-zigzag-list {
        display: flex;
        flex-direction: column;
        gap: 4rem;
    }

    /* ===== Item ===== */
    .services-zigzag-item {
        display: grid;
        grid-template-columns: 1fr 1fr;
        gap: 4rem;
        align-items: center;
        position: relative;
    }

    .services-zigzag-item-reverse {
        direction: rtl;
    }

    .services-zigzag-item-reverse > * {
        direction: ltr;
    }

    /* ===== Content ===== */
    .services-zigzag-content {
        position: relative;
    }

    .services-zigzag-number {
        font-family: var(--font-heading);
        font-size: 4rem;
        font-weight: 900;
        color: var(--color-primary-100);
        line-height: 1;
        margin-bottom: 0.5rem;
    }

    .dark .services-zigzag-number {
        color: var(--color-primary-900);
    }

    .services-zigzag-item-title {
        font-family: var(--font-heading);
        font-size: 1.75rem;
        font-weight: 700;
        color: var(--color-gray-900);
        margin-bottom: 1rem;
    }

    .dark .services-zigzag-item-title {
        color: var(--color-text-primary);
    }

    .services-zigzag-desc {
        font-size: 1rem;
        color: var(--color-gray-600);
        line-height: 1.7;
        margin-bottom: 1.5rem;
    }

    .dark .services-zigzag-desc {
        color: var(--color-text-tertiary);
    }

    /* ===== Features ===== */
    .services-zigzag-features {
        display: flex;
        flex-direction: column;
        gap: 0.75rem;
        margin-bottom: 1.5rem;
    }

    .services-zigzag-feature {
        display: flex;
        align-items: center;
        gap: 0.75rem;
        font-size: 0.9375rem;
        color: var(--color-gray-700);
    }

    .services-zigzag-feature svg {
        color: var(--color-primary-500);
        flex-shrink: 0;
    }

    .dark .services-zigzag-feature {
        color: var(--color-text-secondary);
    }

    .dark .services-zigzag-feature svg {
        color: var(--color-primary-400);
    }

    /* ===== Footer ===== */
    .services-zigzag-footer {
        display: flex;
        align-items: center;
        gap: 1.5rem;
        flex-wrap: wrap;
    }

    .services-zigzag-price {
        font-family: var(--font-heading);
        font-size: 1.125rem;
        font-weight: 700;
        color: var(--color-primary-600);
        padding: 0.5rem 1rem;
        background: var(--color-primary-50);
        border-radius: 0.5rem;
    }

    .dark .services-zigzag-price {
        color: var(--color-primary-400);
        background: rgba(var(--color-primary-rgb, 37, 99, 235), 0.15);
    }

    .services-zigzag-btn {
        display: inline-flex;
        align-items: center;
        gap: 0.5rem;
        padding: 0.75rem 1.5rem;
        background: var(--color-gray-900);
        color: white;
        font-size: 0.875rem;
        font-weight: 600;
        border-radius: 0.75rem;
        transition: all 0.3s ease;
    }

    .services-zigzag-btn:hover {
        background: var(--color-primary-600);
        transform: translateX(4px);
    }

    .dark .services-zigzag-btn {
        background: var(--color-primary-600);
    }

    .dark .services-zigzag-btn:hover {
        background: var(--color-primary-500);
    }

    /* ===== Visual ===== */
    .services-zigzag-visual {
        display: flex;
        align-items: center;
        justify-content: center;
    }

    .services-zigzag-icon-wrapper {
        position: relative;
        width: 200px;
        height: 200px;
        display: flex;
        align-items: center;
        justify-content: center;
    }

    .services-zigzag-icon-bg {
        position: absolute;
        inset: 0;
        background: linear-gradient(135deg, var(--color-primary-100), var(--color-secondary-100));
        border-radius: 2rem;
        transform: rotate(12deg);
        transition: transform 0.4s ease;
    }

    .services-zigzag-item:hover .services-zigzag-icon-bg {
        transform: rotate(-12deg);
    }

    .dark .services-zigzag-icon-bg {
        background: linear-gradient(135deg, var(--color-primary-900), var(--color-secondary-900));
    }

    .services-zigzag-icon {
        position: relative;
        width: 120px;
        height: 120px;
        display: flex;
        align-items: center;
        justify-content: center;
        background: linear-gradient(135deg, var(--color-primary-500), var(--color-secondary-500));
        border-radius: 1.5rem;
        color: white;
        z-index: 2;
        transition: all 0.4s ease;
    }

    .services-zigzag-item:hover .services-zigzag-icon {
        transform: scale(1.1) rotate(-5deg);
    }

    .services-zigzag-icon-ring {
        position: absolute;
        inset: -20px;
        border: 2px dashed var(--color-primary-300);
        border-radius: 2.5rem;
        opacity: 0.5;
        animation: services-zigzag-spin 20s linear infinite;
    }

    .dark .services-zigzag-icon-ring {
        border-color: var(--color-primary-700);
    }

    @keyframes services-zigzag-spin {
        from { transform: rotate(0deg); }
        to { transform: rotate(360deg); }
    }

    .services-zigzag-icon-dots {
        position: absolute;
        inset: 0;
    }

    .services-zigzag-icon-dots span {
        position: absolute;
        width: 10px;
        height: 10px;
        background: var(--color-secondary-400);
        border-radius: 50%;
    }

    .services-zigzag-icon-dots span:nth-child(1) { top: 0; left: 50%; transform: translateX(-50%); }
    .services-zigzag-icon-dots span:nth-child(2) { bottom: 10%; right: 0; }
    .services-zigzag-icon-dots span:nth-child(3) { bottom: 10%; left: 0; background: var(--color-primary-400); }

    /* ===== Connector ===== */
    .services-zigzag-connector {
        position: absolute;
        bottom: -2rem;
        left: 50%;
        width: 2px;
        height: 4rem;
        background: linear-gradient(180deg, var(--color-primary-200), transparent);
        transform: translateX(-50%);
    }

    .dark .services-zigzag-connector {
        background: linear-gradient(180deg, var(--color-primary-800), transparent);
    }

    /* ===== RTL Support ===== */
    [dir="rtl"] .services-zigzag-section {
        direction: rtl;
    }

    [dir="rtl"] .services-zigzag-badge {
        flex-direction: row-reverse;
    }

    [dir="rtl"] .services-zigzag-content {
        text-align: right;
    }

    [dir="rtl"] .services-zigzag-feature {
        flex-direction: row-reverse;
    }

    [dir="rtl"] .services-zigzag-footer {
        justify-content: flex-end;
    }

    [dir="rtl"] .services-zigzag-btn {
        flex-direction: row-reverse;
    }

    [dir="rtl"] .services-zigzag-btn svg {
        transform: scaleX(-1);
    }

    [dir="rtl"] .services-zigzag-btn:hover {
        transform: translateX(-4px);
    }

    [dir="rtl"] .services-zigzag-item-reverse {
        direction: ltr;
    }

    [dir="rtl"] .services-zigzag-item-reverse > * {
        direction: rtl;
    }

    [dir="rtl"] .services-zigzag-item:not(.services-zigzag-item-reverse) {
        direction: rtl;
    }

    [dir="rtl"] .services-zigzag-item:not(.services-zigzag-item-reverse) > * {
        direction: rtl;
    }

    /* ===== Responsive ===== */
    @media (max-width: 768px) {
        .services-zigzag-section {
            padding: 4rem 0;
        }

        .services-zigzag-title {
            font-size: 2rem;
        }

        .services-zigzag-list {
            gap: 3rem;
        }

        .services-zigzag-item,
        .services-zigzag-item-reverse {
            grid-template-columns: 1fr;
            gap: 2rem;
            direction: ltr;
        }

        .services-zigzag-item > *,
        .services-zigzag-item-reverse > * {
            direction: ltr;
        }

        [dir="rtl"] .services-zigzag-item,
        [dir="rtl"] .services-zigzag-item-reverse {
            direction: rtl;
        }

        [dir="rtl"] .services-zigzag-item > *,
        [dir="rtl"] .services-zigzag-item-reverse > * {
            direction: rtl;
        }

        .services-zigzag-visual {
            order: -1;
        }

        .services-zigzag-icon-wrapper {
            width: 160px;
            height: 160px;
        }

        .services-zigzag-icon {
            width: 100px;
            height: 100px;
        }

        .services-zigzag-icon svg {
            width: 2.5rem;
            height: 2.5rem;
        }

        .services-zigzag-number {
            font-size: 3rem;
        }

        .services-zigzag-footer {
            flex-direction: column;
            align-items: flex-start;
        }

        [dir="rtl"] .services-zigzag-footer {
            align-items: flex-end;
        }

        .services-zigzag-btn {
            width: 100%;
            justify-content: center;
        }

        .services-zigzag-connector {
            display: none;
        }
    }
</style>
</div>
