<?php

use Livewire\Component;

new class extends Component {
    public array $section = [];
    public array $items = [];
    public array $company = [];
    public array $social = [];
    public array $services = [];
    public bool $isWhiteBackground = false;
    public ?int $paddingTop = null;
    public ?int $paddingBottom = null;

    // Default static data for services
    private array $defaultItems = [
        [
            'icon' => 'lucide:shirt',
            'title' => 'Wash & Fold',
            'description' => 'Professional washing, drying, and folding service for your everyday clothes.',
            'price' => 'From $2.50/lb',
            'features' => [
                ['text' => 'Same-day service available'],
                ['text' => 'Eco-friendly detergents'],
                ['text' => 'Fabric softener included'],
            ],
            'button_text' => 'Learn More',
            'button_url' => '#contact',
            'button_icon' => 'lucide:arrow-right',
        ],
        [
            'icon' => 'lucide:sparkles',
            'title' => 'Dry Cleaning',
            'description' => 'Expert care for your delicate and special garments that need extra attention.',
            'price' => 'From $8.99/item',
            'features' => [
                ['text' => 'Suits & formal wear'],
                ['text' => 'Delicate fabrics'],
                ['text' => 'Stain treatment included'],
            ],
            'button_text' => 'Learn More',
            'button_url' => '#contact',
            'button_icon' => 'lucide:arrow-right',
        ],
        [
            'icon' => 'lucide:wind',
            'title' => 'Press & Iron',
            'description' => 'Professional pressing and ironing for a crisp, polished look.',
            'price' => 'From $3.50/item',
            'features' => [
                ['text' => 'Steam pressing'],
                ['text' => 'Hand finishing'],
                ['text' => 'Ready to wear'],
            ],
            'button_text' => 'Learn More',
            'button_url' => '#contact',
            'button_icon' => 'lucide:arrow-right',
        ],
        [
            'icon' => 'lucide:bed-double',
            'title' => 'Bedding & Linens',
            'description' => 'Fresh, clean bedding and household linens for maximum comfort.',
            'price' => 'From $15.99/set',
            'features' => [
                ['text' => 'Comforters & duvets'],
                ['text' => 'Sheets & pillowcases'],
                ['text' => 'Allergen removal'],
            ],
            'button_text' => 'Learn More',
            'button_url' => '#contact',
            'button_icon' => 'lucide:arrow-right',
        ],
        [
            'icon' => 'lucide:gem',
            'title' => 'Premium Care',
            'description' => 'Specialized treatment for luxury fabrics and designer pieces.',
            'price' => 'Custom pricing',
            'features' => [
                ['text' => 'Designer labels'],
                ['text' => 'Silk & cashmere'],
                ['text' => 'Hand wash only items'],
            ],
            'button_text' => 'Learn More',
            'button_url' => '#contact',
            'button_icon' => 'lucide:arrow-right',
        ],
        [
            'icon' => 'lucide:zap',
            'title' => 'Express Service',
            'description' => 'Urgent cleaning with quick turnaround when you need it fast.',
            'price' => '+50% rush fee',
            'features' => [
                ['text' => '2-hour turnaround'],
                ['text' => 'Priority handling'],
                ['text' => 'Delivery included'],
            ],
            'button_text' => 'Learn More',
            'button_url' => '#contact',
            'button_icon' => 'lucide:arrow-right',
        ],
    ];

    public function mount(
        array $section = [],
        array $items = [],
        array $company = [],
        array $social = [],
        array $services = [],
        bool $isWhiteBackground = false,
        ?int $paddingTop = null,
        ?int $paddingBottom = null
    ) {
        $this->company = $company;
        $this->social = $social;
        $this->isWhiteBackground = $isWhiteBackground;
        $this->paddingTop = $paddingTop;
        $this->paddingBottom = $paddingBottom;

        // Build section data with defaults
        $this->section = array_merge([
            'badge' => 'Our Services',
            'title' => 'Professional Laundry Services',
            'subtitle' => 'From everyday laundry to delicate dry cleaning, we\'ve got all your garment care needs covered.',
        ], $section);

        // Use items from section content (CMS), then $items prop, then default static data
        $this->items = !empty($section['items']) ? $section['items'] : (!empty($items) ? $items : $this->defaultItems);
    }

    /**
     * Get custom padding style if set
     */
    public function getCustomPaddingStyle(): string
    {
        $styles = [];
        if ($this->paddingTop !== null) {
            $styles[] = "padding-top: {$this->paddingTop}px";
        }
        if ($this->paddingBottom !== null) {
            $styles[] = "padding-bottom: {$this->paddingBottom}px";
        }
        return !empty($styles) ? implode('; ', $styles) : '';
    }
};
?>

<div>
<section id="services" class="services-section {{ $isWhiteBackground ? 'services-white' : 'services-gray' }}" @if($this->getCustomPaddingStyle()) style="{{ $this->getCustomPaddingStyle() }}" @endif>
    {{-- Decorative Background Elements --}}
    <div class="services-bg-decoration">
        <div class="services-blob services-blob-1"></div>
        <div class="services-blob services-blob-2"></div>
    </div>

    <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 relative">
        {{-- Section Header --}}
        <div class="services-header">
            <span class="services-badge">
                <x-icon icon="lucide:sparkles" class="w-4 h-4" />
                <span>{{ $section['badge'] }}</span>
            </span>
            <h2 class="services-title">{{ $section['title'] }}</h2>
            <p class="services-subtitle">{{ $section['subtitle'] }}</p>
        </div>

        {{-- Services Grid --}}
        <div class="services-grid">
            @foreach ($items as $index => $item)
                <div class="service-card scroll-animate" style="animation-delay: {{ $index * 100 }}ms">
                    {{-- Card Glow Effect --}}
                    <div class="service-card-glow"></div>

                    {{-- Card Content --}}
                    <div class="service-card-inner">
                        {{-- Icon --}}
                        <div class="service-icon">
                            <x-icon icon="{{ $item['icon'] ?? 'heroicons:sparkles' }}" class="w-7 h-7" />
                        </div>

                        {{-- Content --}}
                        <div class="service-content">
                            <h3 class="service-title">{{ $item['title'] ?? '' }}</h3>
                            <p class="service-description">{{ $item['description'] ?? '' }}</p>
                        </div>

                        {{-- Price Tag --}}
                        @if (!empty($item['price']))
                            <div class="service-price">
                                <span>{{ $item['price'] }}</span>
                            </div>
                        @endif

                        {{-- Features --}}
                        @if (!empty($item['features']))
                            <ul class="service-features">
                                @foreach ($item['features'] as $feature)
                                    <li class="service-feature">
                                        <span class="service-feature-icon">
                                            <x-icon icon="lucide:check" class="w-3.5 h-3.5" />
                                        </span>
                                        {{-- Support both string format and object format from CMS --}}
                                        <span>{{ is_array($feature) ? ($feature['text'] ?? '') : $feature }}</span>
                                    </li>
                                @endforeach
                            </ul>
                        @endif

                        {{-- Action Button --}}
                        @if (!empty($item['button_text']) && !empty($item['button_url']))
                            <a href="{{ $item['button_url'] }}" class="service-button">
                                <span>{{ $item['button_text'] }}</span>
                                <x-icon icon="{{ $item['button_icon'] ?? 'lucide:arrow-right' }}" class="w-4 h-4" />
                            </a>
                        @endif
                    </div>
                </div>
            @endforeach
        </div>
    </div>
</section>

<style>
    /* ===== Services Section Base ===== */
    .services-section {
        position: relative;
        padding: 6rem 0;
        overflow: hidden;
    }

    .services-white {
        background: #ffffff;
    }

    .services-gray {
        background: linear-gradient(180deg, #f8fafc 0%, #f1f5f9 100%);
    }

    .dark .services-white {
        background: var(--color-bg-secondary);
    }

    .dark .services-gray {
        background: linear-gradient(180deg, var(--color-bg-primary) 0%, var(--color-bg-secondary) 100%);
    }

    /* ===== Background Decoration ===== */
    .services-bg-decoration {
        position: absolute;
        inset: 0;
        overflow: hidden;
        pointer-events: none;
    }

    .services-blob {
        position: absolute;
        border-radius: 50%;
        filter: blur(80px);
        opacity: 0.4;
    }

    .services-blob-1 {
        width: 500px;
        height: 500px;
        background: linear-gradient(135deg, var(--color-primary-200), var(--color-secondary-200));
        top: -200px;
        right: -200px;
    }

    .services-blob-2 {
        width: 400px;
        height: 400px;
        background: linear-gradient(135deg, var(--color-secondary-200), var(--color-primary-200));
        bottom: -150px;
        left: -150px;
    }

    .dark .services-blob {
        opacity: 0.15;
    }

    .dark .services-blob-1 {
        background: linear-gradient(135deg, var(--color-primary-600), var(--color-secondary-600));
    }

    .dark .services-blob-2 {
        background: linear-gradient(135deg, var(--color-secondary-600), var(--color-primary-600));
    }

    /* ===== Section Header ===== */
    .services-header {
        text-align: center;
        margin-bottom: 4rem;
    }

    .services-badge {
        display: inline-flex;
        align-items: center;
        gap: 0.5rem;
        padding: 0.5rem 1rem;
        background: linear-gradient(135deg, var(--color-primary-50), var(--color-secondary-50));
        border: 1px solid var(--color-primary-100);
        border-radius: 9999px;
        font-size: 0.875rem;
        font-weight: 600;
        color: var(--color-primary-700);
        margin-bottom: 1.5rem;
    }

    .dark .services-badge {
        background: linear-gradient(135deg, rgba(var(--color-primary-600), 0.2), rgba(var(--color-secondary-600), 0.2));
        background: var(--color-bg-tertiary);
        border-color: var(--color-border);
        color: var(--color-primary-400);
    }

    .services-title {
        font-family: var(--font-heading);
        font-size: 2.5rem;
        font-weight: 800;
        color: var(--color-gray-900);
        margin-bottom: 1rem;
        line-height: 1.2;
    }

    .dark .services-title {
        color: var(--color-text-primary);
    }

    @media (min-width: 768px) {
        .services-title {
            font-size: 3rem;
        }
    }

    .services-subtitle {
        font-size: 1.125rem;
        color: var(--color-gray-600);
        max-width: 600px;
        margin: 0 auto;
        line-height: 1.7;
    }

    .dark .services-subtitle {
        color: var(--color-text-tertiary);
    }

    /* ===== Services Grid ===== */
    .services-grid {
        display: grid;
        gap: 1.5rem;
    }

    @media (min-width: 640px) {
        .services-grid {
            grid-template-columns: repeat(2, 1fr);
        }
    }

    @media (min-width: 1024px) {
        .services-grid {
            grid-template-columns: repeat(3, 1fr);
            gap: 2rem;
        }
    }

    /* ===== Service Card ===== */
    .service-card {
        position: relative;
        border-radius: 1.25rem;
        transition: transform 0.4s cubic-bezier(0.4, 0, 0.2, 1);
    }

    .service-card:hover {
        transform: translateY(-8px);
    }

    .service-card-glow {
        position: absolute;
        inset: 0;
        border-radius: 1.25rem;
        opacity: 0;
        transition: opacity 0.4s ease;
        filter: blur(20px);
        transform: translateY(10px);
    }

    .service-card:hover .service-card-glow {
        opacity: 0.3;
    }

    .dark .service-card:hover .service-card-glow {
        opacity: 0.15;
    }

    .service-card-glow {
        background: var(--color-primary-400);
    }

    .service-card-inner {
        position: relative;
        background: #ffffff;
        border: 1px solid var(--color-gray-200);
        border-radius: 1.25rem;
        padding: 1.75rem;
        height: 100%;
        display: flex;
        flex-direction: column;
        transition: border-color 0.3s ease, box-shadow 0.3s ease;
    }

    .dark .service-card-inner {
        background: var(--color-card-bg);
        border-color: var(--color-border);
    }

    .service-card:hover .service-card-inner {
        border-color: var(--color-primary-200);
        box-shadow: 0 20px 40px -12px rgba(0, 0, 0, 0.1);
    }

    .dark .service-card:hover .service-card-inner {
        border-color: var(--color-primary-600);
        box-shadow: 0 20px 40px -12px rgba(0, 0, 0, 0.4);
    }

    /* ===== Service Icon ===== */
    .service-icon {
        width: 3.5rem;
        height: 3.5rem;
        border-radius: 1rem;
        display: flex;
        align-items: center;
        justify-content: center;
        margin-bottom: 1.25rem;
        transition: transform 0.3s ease;
    }

    .service-card:hover .service-icon {
        transform: scale(1.1) rotate(-5deg);
    }

    .service-icon {
        background: linear-gradient(135deg, var(--color-primary-100), var(--color-primary-50));
        color: var(--color-primary-600);
    }

    .dark .service-icon {
        background: linear-gradient(135deg, rgba(var(--color-primary-rgb, 37, 99, 235), 0.2), rgba(var(--color-primary-rgb, 37, 99, 235), 0.1));
        color: var(--color-primary-400);
    }

    /* ===== Service Content ===== */
    .service-content {
        flex: 1;
        margin-bottom: 1rem;
    }

    .service-title {
        font-family: var(--font-heading);
        font-size: 1.25rem;
        font-weight: 700;
        color: var(--color-gray-900);
        margin-bottom: 0.5rem;
    }

    .dark .service-title {
        color: var(--color-text-primary);
    }

    .service-description {
        font-size: 0.9375rem;
        color: var(--color-gray-600);
        line-height: 1.6;
    }

    .dark .service-description {
        color: var(--color-text-tertiary);
    }

    /* ===== Service Price ===== */
    .service-price {
        display: inline-flex;
        padding: 0.375rem 0.875rem;
        border-radius: 9999px;
        font-size: 0.875rem;
        font-weight: 600;
        margin-bottom: 1rem;
        background: var(--color-primary-50);
        color: var(--color-primary-700);
    }

    .dark .service-price {
        background: rgba(var(--color-primary-rgb, 37, 99, 235), 0.15);
        color: var(--color-primary-400);
    }

    /* ===== Service Features ===== */
    .service-features {
        display: flex;
        flex-direction: column;
        gap: 0.625rem;
    }

    .service-feature {
        display: flex;
        align-items: center;
        gap: 0.625rem;
        font-size: 0.875rem;
        color: var(--color-gray-600);
    }

    .dark .service-feature {
        color: var(--color-text-tertiary);
    }

    .service-feature-icon {
        width: 1.25rem;
        height: 1.25rem;
        border-radius: 50%;
        display: flex;
        align-items: center;
        justify-content: center;
        flex-shrink: 0;
        background: var(--color-primary-100);
        color: var(--color-primary-600);
    }

    .dark .service-feature-icon {
        background: rgba(var(--color-primary-rgb, 37, 99, 235), 0.2);
        color: var(--color-primary-400);
    }

    /* ===== Service Button ===== */
    .service-button {
        display: inline-flex;
        align-items: center;
        gap: 0.5rem;
        margin-top: 1rem;
        padding: 0.625rem 1.25rem;
        background: linear-gradient(135deg, var(--color-primary-600), var(--color-primary-700));
        color: white;
        font-size: 0.875rem;
        font-weight: 600;
        border-radius: 0.5rem;
        transition: all 0.3s ease;
    }

    .service-button:hover {
        background: linear-gradient(135deg, var(--color-primary-700), var(--color-primary-800));
        transform: translateY(-2px);
        box-shadow: 0 4px 12px rgba(var(--color-primary-rgb, 37, 99, 235), 0.35);
    }

    .dark .service-button {
        background: linear-gradient(135deg, var(--color-primary-500), var(--color-primary-600));
    }

    .dark .service-button:hover {
        background: linear-gradient(135deg, var(--color-primary-400), var(--color-primary-500));
    }

    /* ===== RTL Support ===== */
    [dir="rtl"] .services-section {
        direction: rtl;
    }

    [dir="rtl"] .services-blob-1 {
        right: auto;
        left: -200px;
    }

    [dir="rtl"] .services-blob-2 {
        left: auto;
        right: -150px;
    }

    [dir="rtl"] .services-badge {
        flex-direction: row-reverse;
    }

    [dir="rtl"] .service-card-inner {
        text-align: right;
    }

    [dir="rtl"] .service-content {
        text-align: right;
    }

    [dir="rtl"] .service-features {
        align-items: flex-start;
    }

    [dir="rtl"] .service-feature {
        flex-direction: row-reverse;
        text-align: right;
    }

    [dir="rtl"] .service-button {
        flex-direction: row-reverse;
    }

    [dir="rtl"] .service-button svg {
        transform: scaleX(-1);
    }

    [dir="rtl"] .service-card:hover .service-icon {
        transform: scale(1.1) rotate(5deg);
    }

    /* ===== Responsive ===== */
    @media (max-width: 640px) {
        .services-section {
            padding: 4rem 0;
        }

        .services-header {
            margin-bottom: 3rem;
        }

        .services-title {
            font-size: 1.875rem;
        }

        .services-subtitle {
            font-size: 1rem;
        }

        .service-card-inner {
            padding: 1.5rem;
        }

        .service-icon {
            width: 3rem;
            height: 3rem;
        }

        .service-button {
            width: 100%;
            justify-content: center;
        }
    }
</style>
</div>
