<?php

use Livewire\Component;

new class extends Component {
    public array $section = [];
    public array $items = [];
    public array $company = [];
    public array $social = [];
    public array $services = [];
    public bool $isWhiteBackground = false;
    public ?int $paddingTop = null;
    public ?int $paddingBottom = null;

    public function mount(
        array $section = [],
        array $items = [],
        array $company = [],
        array $social = [],
        array $services = [],
        bool $isWhiteBackground = false,
        ?int $paddingTop = null,
        ?int $paddingBottom = null
    ) {
        $this->company = $company;
        $this->social = $social;
        $this->services = $services;
        $this->isWhiteBackground = $isWhiteBackground;
        $this->items = $items;
        $this->paddingTop = $paddingTop;
        $this->paddingBottom = $paddingBottom;

        $this->section = array_merge([
            'badge' => 'Watch Video',
            'title' => 'See How We Deliver Excellence',
            'subtitle' => 'A glimpse into our process and commitment to quality',
            'description' => 'Watch our story and discover why thousands of customers trust us with their needs. Our dedicated team works tirelessly to ensure your complete satisfaction.',
            'video_url' => 'https://www.youtube.com/embed/dQw4w9WgXcQ',
            'video_thumbnail' => 'https://images.unsplash.com/photo-1522202176988-66273c2fd55f?w=800&h=450&fit=crop',
        ], $section);
    }

    public function getCustomPaddingStyle(): string
    {
        $styles = [];
        if ($this->paddingTop !== null) {
            $styles[] = "padding-top: {$this->paddingTop}px";
        }
        if ($this->paddingBottom !== null) {
            $styles[] = "padding-bottom: {$this->paddingBottom}px";
        }
        return !empty($styles) ? implode('; ', $styles) : '';
    }
};
?>

<div>
<section id="video" class="section {{ $isWhiteBackground ? 'bg-white dark:bg-gray-800' : 'bg-gray-50 dark:bg-gray-900' }}" @if($this->getCustomPaddingStyle()) style="{{ $this->getCustomPaddingStyle() }}" @endif>
    <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        <div class="grid lg:grid-cols-2 gap-12 items-center">
            {{-- Content Column --}}
            <div class="scroll-animate" data-animation="fade-right">
                <span class="badge badge-primary mb-4">{{ $section['badge'] }}</span>

                <h2 class="section-title text-left">{{ $section['title'] }}</h2>

                @if(!empty($section['subtitle']))
                <p class="text-lg text-primary-600 dark:text-primary-400 font-medium mb-4">
                    {{ $section['subtitle'] }}
                </p>
                @endif

                @if(!empty($section['description']))
                <div class="prose prose-lg dark:prose-invert max-w-none text-gray-600 dark:text-gray-300">
                    {!! $section['description'] !!}
                </div>
                @endif
            </div>

            {{-- Video Column --}}
            <div class="scroll-animate" data-animation="fade-left">
                <div class="video-wrapper aspect-video shadow-2xl" x-data="{ open: false }">
                    @if(!empty($section['video_thumbnail']))
                    <img src="{{ $section['video_thumbnail'] }}" alt="{{ $section['title'] }}" class="w-full h-full object-cover">
                    @else
                    <div class="w-full h-full bg-gradient-to-br from-primary-500 to-primary-700"></div>
                    @endif

                    <div class="video-overlay">
                        <button @click="open = true" class="play-button" aria-label="{{ __('Play Video') }}">
                            <svg class="w-8 h-8 text-white" fill="currentColor" viewBox="0 0 24 24">
                                <path d="M8 5v14l11-7z"/>
                            </svg>
                        </button>
                    </div>

                    {{-- Video Modal --}}
                    <div class="video-modal" :class="{ 'active': open }" @click.self="open = false" @keydown.escape.window="open = false">
                        <div class="video-modal-content">
                            <button @click="open = false" class="video-modal-close" aria-label="{{ __('Close') }}">
                                <svg class="w-8 h-8" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"/>
                                </svg>
                            </button>
                            @if(!empty($section['video_url']))
                            <template x-if="open">
                                <iframe
                                    src="{{ $section['video_url'] }}"
                                    class="w-full h-full rounded-xl"
                                    frameborder="0"
                                    allow="accelerometer; autoplay; clipboard-write; encrypted-media; gyroscope; picture-in-picture"
                                    allowfullscreen>
                                </iframe>
                            </template>
                            @endif
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</section>

<style>
    .video-wrapper {
        position: relative;
        overflow: hidden;
        border-radius: 1rem;
    }
    .video-overlay {
        position: absolute;
        inset: 0;
        background: linear-gradient(to bottom, rgba(0,0,0,0.2), rgba(0,0,0,0.5));
        display: flex;
        align-items: center;
        justify-content: center;
        transition: background 0.3s ease;
    }
    .video-wrapper:hover .video-overlay {
        background: linear-gradient(to bottom, rgba(0,0,0,0.3), rgba(0,0,0,0.6));
    }
    .play-button {
        width: 80px;
        height: 80px;
        background: var(--color-primary-500);
        border-radius: 50%;
        display: flex;
        align-items: center;
        justify-content: center;
        cursor: pointer;
        transition: transform 0.3s ease, box-shadow 0.3s ease;
        box-shadow: 0 0 0 0 rgba(var(--color-primary-500-rgb), 0.4);
        animation: pulse-play 2s infinite;
    }
    .play-button:hover {
        transform: scale(1.1);
    }
    @keyframes pulse-play {
        0% {
            box-shadow: 0 0 0 0 rgba(var(--color-primary-500-rgb), 0.4);
        }
        70% {
            box-shadow: 0 0 0 20px rgba(var(--color-primary-500-rgb), 0);
        }
        100% {
            box-shadow: 0 0 0 0 rgba(var(--color-primary-500-rgb), 0);
        }
    }
    .play-button svg {
        margin-left: 4px;
    }
    [dir="rtl"] .play-button svg {
        margin-left: 0;
        margin-right: 4px;
    }
    .video-modal {
        display: none;
        position: fixed;
        inset: 0;
        z-index: 9999;
        background: rgba(0,0,0,0.9);
        align-items: center;
        justify-content: center;
        padding: 2rem;
    }
    .video-modal.active {
        display: flex;
    }
    .video-modal-content {
        position: relative;
        width: 100%;
        max-width: 900px;
        aspect-ratio: 16/9;
    }
    .video-modal-close {
        position: absolute;
        top: -40px;
        right: 0;
        color: white;
        cursor: pointer;
        font-size: 1.5rem;
    }
    [dir="rtl"] .video-modal-close {
        right: auto;
        left: 0;
    }
</style>
</div>
