<?php

use App\Livewire\LivewireBaseComponent;
use App\Models\WebsiteBuilder;
use App\Models\WebsitePage;

/**
 * Homepage Livewire Component
 *
 * Extends LivewireBaseComponent to inherit common data (theme, seo, company, nav, social, footer, services)
 * Dynamically renders sections from database based on sort_order and website_page_id
 *
 * URL patterns:
 * - /{slug} or / (for primary website) -> page_key = 'home'
 * - /{slug}/{pageKey} or /{pageKey} (for primary website) -> page_key from URL
 *
 * All section components receive a standardized props structure:
 * - section: Contains title, subtitle, and all content fields from database
 * - items: Array of items from content.items (if exists)
 * - company: Company data (available to all sections)
 * - social: Social links (available to all sections)
 * - services: Services data (available to all sections)
 */
new class extends LivewireBaseComponent {
    // All sections from database (ordered by sort_order)
    public \Illuminate\Support\Collection $dbSections;

    // Current page key (e.g., 'home', 'about-us', etc.)
    public ?string $pageKey = null;

    // Current website page
    public ?WebsitePage $websitePage = null;

    /**
     * Mount the component
     *
     * @param string|null $slug Website slug for /w/{slug} routes
     * @param string|null $pageKey Page key from URL (e.g., 'about-us', 'services')
     */
    public function mount($slug = null, $pageKey = null)
    {
        // Initialize common data (theme, seo, company, nav, social, footer, services)
        parent::mount($slug);

        // Determine page key from:
        // 1. Direct parameter (passed from route)
        // 2. Route parameter from request
        // 3. Shared view data (legacy support)
        // 4. Default to 'home'
        $this->pageKey = $pageKey
            ?? request()->route('pageKey')
            ?? view()->shared('frontPageKey')
            ?? 'home';

        // Load website page if we have a website
        if ($this->frontWebsite) {
            $this->websitePage = WebsitePage::withoutGlobalScopes()
                ->where('website_id', $this->frontWebsite->id)
                ->where('page_key', $this->pageKey)
                ->first();
        }

        // Load all sections from database ordered by sort_order
        $this->loadSections();
    }

    /**
     * Load all sections from database ordered by sort_order
     * Filters by website_page_id based on the current page_key
     */
    protected function loadSections(): void
    {
        try {
            if ($this->frontWebsite) {
                // Build base query with website and language filter
                $query = WebsiteBuilder::withoutGlobalScopes()
                    ->where('website_id', $this->frontWebsite->id)
                    ->where('website_language_id', $this->currentLangId);

                // Filter by website_page_id if we have a website page
                if ($this->websitePage) {
                    $query->where('website_page_id', $this->websitePage->id);
                }

                $this->dbSections = $query->orderBy('sort_order', 'asc')->get();

                // Fallback to English if empty
                if ($this->dbSections->isEmpty() && $this->currentLangKey !== 'en') {
                    $enLang = \App\Models\WebsiteLanguage::withoutGlobalScopes()
                        ->where('key', 'en')
                        ->where('company_id', $this->frontWebsite->company_id)
                        ->first();

                    $query = WebsiteBuilder::withoutGlobalScopes()
                        ->where('website_id', $this->frontWebsite->id)
                        ->where('website_language_id', $enLang?->id);

                    if ($this->websitePage) {
                        $query->where('website_page_id', $this->websitePage->id);
                    }

                    $this->dbSections = $query->orderBy('sort_order', 'asc')->get();
                }

                // Final fallback without lang filter (but still with website_page_id)
                if ($this->dbSections->isEmpty()) {
                    $query = WebsiteBuilder::withoutGlobalScopes()
                        ->where('website_id', $this->frontWebsite->id);

                    if ($this->websitePage) {
                        $query->where('website_page_id', $this->websitePage->id);
                    }

                    $this->dbSections = $query->orderBy('sort_order', 'asc')->get();
                }
            } else {
                // Company-level settings with language filter
                $this->dbSections = WebsiteBuilder::where('website_language_id', $this->currentLangId)->orderBy('sort_order', 'asc')->get();

                // Fallback to English if empty
                if ($this->dbSections->isEmpty() && $this->currentLangKey !== 'en') {
                    $enLang = \App\Models\WebsiteLanguage::where('key', 'en')->first();
                    $this->dbSections = WebsiteBuilder::where('website_language_id', $enLang?->id)->orderBy('sort_order', 'asc')->get();
                }

                // Final fallback without lang filter
                if ($this->dbSections->isEmpty()) {
                    $this->dbSections = WebsiteBuilder::orderBy('sort_order', 'asc')->get();
                }
            }
        } catch (\Exception $e) {
            $this->dbSections = collect();
        }
    }

    /**
     * Get component name for a section
     */
    public function getComponentName(WebsiteBuilder $section): string
    {
        $sectionKey = $section->section_key;
        $layout = $section->section_layout ?? $sectionKey;

        return $this->themeName . '::components.' . $sectionKey . '.' . $layout;
    }

    /**
     * Get standardized props for any section component
     * All components receive the same structure - they extract what they need
     */
    public function getSectionProps(WebsiteBuilder $section): array
    {
        // Ensure content is an array (handle JSON string case)
        $content = $section->content_with_urls ?? ($section->content ?? []);
        if (is_string($content)) {
            $content = json_decode($content, true) ?? [];
        }
        if (!is_array($content)) {
            $content = [];
        }

        // Ensure items is always an array
        $items = $content['items'] ?? [];
        if (is_string($items)) {
            $items = json_decode($items, true) ?? [];
        }
        if (!is_array($items)) {
            $items = [];
        }

        // Build the standardized section data
        // Merges database fields with content fields
        $sectionData = array_merge(
            [
                'section_key' => $section->section_key,
                'section_layout' => $section->section_layout ?? $section->section_key,
                'title' => $section->title,
                'subtitle' => $section->subtitle,
            ],
            $content, // All content fields (badge, description, cta_primary, etc.)
        );

        // Return standardized props - all components receive the same structure
        return [
            'section' => $sectionData,
            'items' => $items,
            'company' => $this->company,
            'nav' => $this->nav,
            'navItems' => $this->navItems,
            'social' => $this->social,
            'footer' => $this->footer, // Footer defaults from LivewireBaseComponent
            'services' => $this->services,
            'languages' => $this->languages,
            'currentLangKey' => $this->currentLangKey,
            'websiteSlug' => $this->websiteSlug,
            'isWhiteBackground' => $section->is_white_background ?? false,
            'paddingTop' => $section->padding_top,
            'paddingBottom' => $section->padding_bottom,
        ];
    }
};
?>

<div>
    @foreach ($dbSections as $section)
        @livewire($this->getComponentName($section), $this->getSectionProps($section), key($section->id))
    @endforeach
</div>
