<?php

use App\Livewire\LivewireBaseComponent;
use App\Models\WebsiteBuilder;
use App\Models\ThemeSection;
use Illuminate\Support\Facades\File;

/**
 * Section Preview Livewire Component
 *
 * Used to preview a single section layout in an iframe.
 * This allows the admin panel to show live previews of different section layouts.
 *
 * URL: /section-preview/{sectionKey}/{layoutName}?website_id={xid}
 */
new class extends LivewireBaseComponent {
    public string $sectionKey = '';
    public string $layoutName = '';
    public array $sectionData = [];

    /**
     * Mount the component
     *
     * @param string|null $slug Not used - kept for parent compatibility
     * @param string $sectionKey The section key (e.g., 'hero', 'services', 'pricing')
     * @param string $layoutName The layout name (e.g., 'hero', 'hero-split', 'hero-fullscreen')
     */
    public function mount($slug = null, string $sectionKey = '', string $layoutName = '')
    {
        $this->sectionKey = $sectionKey;
        $this->layoutName = $layoutName ?: $sectionKey;

        // Get website from query parameter
        $websiteXid = request()->query('website_id');
        if ($websiteXid) {
            $websiteId = \App\Classes\Common::getIdFromHash($websiteXid);
            $this->frontWebsite = \App\Models\Website::find($websiteId);
            $this->websiteSlug = $this->frontWebsite?->slug;
        }

        // Initialize common data (theme, seo, company, nav, social, footer, services)
        parent::mount($this->websiteSlug);

        // Force light mode if requested (for admin panel preview consistency)
        if (request()->query('force_light_mode')) {
            $this->theme['mode'] = 'light';
        }

        // Load section data
        $this->loadSectionData();
    }

    /**
     * Load section data from database or theme section defaults
     */
    protected function loadSectionData(): void
    {
        // First try to get from website_builder for this website
        $section = null;

        if ($this->frontWebsite) {
            $section = WebsiteBuilder::withoutGlobalScopes()
                ->where('website_id', $this->frontWebsite->id)
                ->where('section_key', $this->sectionKey)
                ->where('website_language_id', $this->currentLangId)
                ->first();
        }

        if ($section) {
            // Use existing section data
            $content = $section->content_with_urls ?? ($section->content ?? []);
            if (is_string($content)) {
                $content = json_decode($content, true) ?? [];
            }

            $this->sectionData = array_merge(
                [
                    'section_key' => $section->section_key,
                    'section_layout' => $this->layoutName,
                    'title' => $section->title,
                    'subtitle' => $section->subtitle,
                ],
                $content
            );
        } else {
            // Fall back to theme section defaults
            $themeSection = ThemeSection::where('section_key', $this->sectionKey)
                ->where('is_active', true)
                ->first();

            if ($themeSection) {
                $content = $themeSection->default_content ?? [];
                if (is_string($content)) {
                    $content = json_decode($content, true) ?? [];
                }

                $this->sectionData = array_merge(
                    [
                        'section_key' => $this->sectionKey,
                        'section_layout' => $this->layoutName,
                        'title' => $themeSection->default_title ?? $themeSection->name,
                        'subtitle' => $themeSection->default_subtitle ?? '',
                    ],
                    $content
                );
            } else {
                // Minimal fallback data
                $this->sectionData = [
                    'section_key' => $this->sectionKey,
                    'section_layout' => $this->layoutName,
                    'title' => ucwords(str_replace(['_', '-'], ' ', $this->sectionKey)),
                    'subtitle' => 'Preview section',
                ];
            }
        }
    }

    /**
     * Get component name for the section
     */
    public function getComponentName(): string
    {
        return $this->themeName . '::components.' . $this->sectionKey . '.' . $this->layoutName;
    }

    /**
     * Check if the component view file exists
     */
    public function componentExists(): bool
    {
        $viewPath = resource_path("views/themes/{$this->themeName}/components/{$this->sectionKey}/⚡{$this->layoutName}.blade.php");
        return File::exists($viewPath);
    }

    /**
     * Get standardized props for the section component
     */
    public function getSectionProps(): array
    {
        $items = $this->sectionData['items'] ?? [];
        if (is_string($items)) {
            $items = json_decode($items, true) ?? [];
        }

        return [
            'section' => $this->sectionData,
            'items' => $items,
            'company' => $this->company,
            'nav' => $this->nav,
            'navItems' => $this->navItems,
            'social' => $this->social,
            'footer' => $this->footer, // Footer defaults from LivewireBaseComponent
            'services' => $this->services,
            'languages' => $this->languages,
            'currentLangKey' => $this->currentLangKey,
            'websiteSlug' => $this->websiteSlug,
            'isWhiteBackground' => false,
            'paddingTop' => null,
            'paddingBottom' => null,
        ];
    }
};
?>

<div>
<div class="section-preview-wrapper">
    @if ($this->componentExists())
        @livewire($this->getComponentName(), $this->getSectionProps(), key($sectionKey . '-' . $layoutName))
    @else
        <div class="section-preview-error">
            <div class="error-content">
                <svg xmlns="http://www.w3.org/2000/svg" class="error-icon" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-3L13.732 4c-.77-1.333-2.694-1.333-3.464 0L3.34 16c-.77 1.333.192 3 1.732 3z" />
                </svg>
                <h3>Layout Not Found</h3>
                <p>The layout <strong>{{ $layoutName }}</strong> for section <strong>{{ $sectionKey }}</strong> does not exist.</p>
            </div>
        </div>
    @endif
</div>

<style>
    .section-preview-wrapper {
        min-height: 100vh;
        background: #fff;
    }

    .section-preview-error {
        display: flex;
        align-items: center;
        justify-content: center;
        min-height: 300px;
        padding: 2rem;
        background: linear-gradient(135deg, #fef2f2 0%, #fee2e2 100%);
    }

    .error-content {
        text-align: center;
        max-width: 400px;
    }

    .error-icon {
        width: 64px;
        height: 64px;
        color: #ef4444;
        margin: 0 auto 1rem;
    }

    .error-content h3 {
        font-size: 1.25rem;
        font-weight: 600;
        color: #991b1b;
        margin-bottom: 0.5rem;
    }

    .error-content p {
        color: #b91c1c;
        font-size: 0.875rem;
    }

    .error-content strong {
        font-weight: 600;
    }
</style>
</div>
