<?php

use Illuminate\Support\Facades\Route;
use App\Models\Website;
use App\Models\WebsiteSetting;
use App\Models\WebsiteLanguage;

/**
 * Prepare front website data and share with views.
 * Common function used by all front routes.
 *
 * @param Website $website
 * @param string $slug
 * @param bool $checkQueryLang Whether to check for lang query parameter
 * @return string Theme name for the component
 */
function prepareFrontWebsiteData(Website $website, string $slug, bool $checkQueryLang = false): string
{
    $langKey = null;
    $langId = null;

    // Check for lang query parameter first (for page routes)
    if ($checkQueryLang) {
        $queryLang = request()->query('lang');
        if ($queryLang) {
            $lang = WebsiteLanguage::withoutGlobalScopes()
                ->where('key', $queryLang)
                ->where('website_id', $website->id)
                ->first();
            if ($lang) {
                $langKey = $lang->key;
                $langId = $lang->id;
                session(['website_lang' => $langKey]);
            }
        }
    }

    // If no lang set yet, check slug-based cookie (same pattern as theme mode)
    if (!$langKey) {
        // Cookie key format: lang_{slug} (e.g., lang_sparklewash, lang_default)
        $langCookieKey = 'lang_' . $slug;
        $userLangKey = request()->cookie($langCookieKey);

        if ($userLangKey) {
            // Check if this language exists for the website
            $lang = WebsiteLanguage::withoutGlobalScopes()
                ->where('key', $userLangKey)
                ->where('website_id', $website->id)
                ->first();
            $langKey = $lang?->key;
            $langId = $lang?->id;
        }

        // If no valid language found, use the default language for this website
        if (!$langKey) {
            $defaultLang = WebsiteLanguage::withoutGlobalScopes()
                ->where('website_id', $website->id)
                ->where('is_default', true)
                ->first();

            if ($defaultLang) {
                $langKey = $defaultLang->key;
                $langId = $defaultLang->id;
            } else {
                // Fallback to English or first available language for this website
                $enLang = WebsiteLanguage::withoutGlobalScopes()
                    ->where('website_id', $website->id)
                    ->where('key', 'en')
                    ->first();

                if ($enLang) {
                    $langKey = 'en';
                    $langId = $enLang->id;
                } else {
                    // Get first available language for this website
                    $firstLang = WebsiteLanguage::withoutGlobalScopes()
                        ->where('website_id', $website->id)
                        ->first();
                    $langKey = $firstLang?->key ?? 'en';
                    $langId = $firstLang?->id;
                }
            }
        }
    }

    // Get website settings for the determined language
    $websiteSettings = WebsiteSetting::withoutGlobalScopes()
        ->where('website_id', $website->id)
        ->where('website_language_id', $langId)
        ->first();

    // Fallback to English if not found for selected language
    if (!$websiteSettings && $langKey !== 'en') {
        $enLang = WebsiteLanguage::withoutGlobalScopes()
            ->where('key', 'en')
            ->where('website_id', $website->id)
            ->first();
        $websiteSettings = WebsiteSetting::withoutGlobalScopes()
            ->where('website_id', $website->id)
            ->where('website_language_id', $enLang?->id)
            ->first();
    }

    // Get theme name from website table, default to 'tailwind_theme'
    $themeName = $website->theme_name ?? 'tailwind_theme';

    // Share data with views
    view()->share('frontWebsite', $website);
    view()->share('frontWebsiteSettings', $websiteSettings);
    view()->share('frontThemeName', $themeName);
    view()->share('frontWebsiteSlug', $slug);
    view()->share('frontLangKey', $langKey);
    view()->share('frontLangId', $langId);

    return $themeName;
}

// Get the website route prefix from company settings
$websitePrefix = website_route_prefix();

// Root URL Route - Only for non-saas when a primary website is set
if (app_type() == 'non-saas' && \Illuminate\Support\Facades\Schema::hasTable('websites')) {
    // Check if primary website exists
    $primaryWebsite = Website::withoutGlobalScopes()
        ->where('is_primary', true)
        ->first();

    if ($primaryWebsite) {
        Route::dynamicLivewire('/', function () {
            $website = Website::withoutGlobalScopes()
                ->where('is_primary', true)
                ->first();

            $themeName = prepareFrontWebsiteData($website, $website->slug);

            // Share home as the default page key
            view()->share('frontPageKey', 'home');

            return "{$themeName}::pages.homepage";
        })->name('landing.root');

        // Root URL Website Page Route - for primary website custom pages (e.g., /about-us, /services)
        Route::dynamicLivewire('/{pageKey}', function (string $pageKey) {
            $website = Website::withoutGlobalScopes()
                ->where('is_primary', true)
                ->first();

            // Verify the page_key exists for this website
            $websitePage = \App\Models\WebsitePage::withoutGlobalScopes()
                ->where('website_id', $website->id)
                ->where('page_key', $pageKey)
                ->first();

            if (!$websitePage) {
                abort(404);
            }

            $themeName = prepareFrontWebsiteData($website, $website->slug);

            // Share the page key for the homepage component
            view()->share('frontPageKey', $pageKey);

            return "{$themeName}::pages.homepage";
        })->name('landing.root.website-page')->where('pageKey', '[a-z0-9\-]+');

        // Root URL Page Route - for primary website static pages (Privacy Policy, Terms, etc.)
        Route::dynamicLivewire('/page/{pageSlug}', function (string $pageSlug) {
            $website = Website::withoutGlobalScopes()
                ->where('is_primary', true)
                ->first();

            $themeName = prepareFrontWebsiteData($website, $website->slug, true);

            return "{$themeName}::pages.page";
        })->name('landing.root.page');
    } else {
        // Redirect to admin login if no primary website or no website builders
        Route::get('/', function () {
            return redirect('/admin/login');
        })->name('landing.root');

        Route::get('/{pageKey}', function () {
            return redirect('/admin/login');
        })->name('landing.root.website-page')->where('pageKey', '[a-z0-9\-]+');

        Route::get('/page/{pageSlug}', function () {
            return redirect('/admin/login');
        })->name('landing.root.page');
    }
}

// Website Landing Page - Dynamic Theme Based on Slug
Route::dynamicLivewire('/' . $websitePrefix . '/{slug}', function (string $slug) {
    $website = Website::withoutGlobalScopes()
        ->where('slug', $slug)
        ->firstOrFail();

    $themeName = prepareFrontWebsiteData($website, $slug);

    // Share home as the default page key
    view()->share('frontPageKey', 'home');

    return "{$themeName}::pages.homepage";
})->name('landing.home');

// Website Page Route with page_key - for custom website pages (e.g., /w/my-website/about-us)
Route::dynamicLivewire('/' . $websitePrefix . '/{slug}/{pageKey}', function (string $slug, string $pageKey) {
    $website = Website::withoutGlobalScopes()
        ->where('slug', $slug)
        ->firstOrFail();

    // Verify the page_key exists for this website
    $websitePage = \App\Models\WebsitePage::withoutGlobalScopes()
        ->where('website_id', $website->id)
        ->where('page_key', $pageKey)
        ->first();

    if (!$websitePage) {
        abort(404);
    }

    $themeName = prepareFrontWebsiteData($website, $slug);

    // Share the page key for the homepage component
    view()->share('frontPageKey', $pageKey);

    return "{$themeName}::pages.homepage";
})->name('landing.website-page')->where('pageKey', '[a-z0-9\-]+');

// Website Page Route - Displays individual front pages (Privacy Policy, Terms, etc.)
Route::dynamicLivewire('/' . $websitePrefix . '/{slug}/page/{pageSlug}', function (string $slug, string $pageSlug) {
    $website = Website::withoutGlobalScopes()
        ->where('slug', $slug)
        ->firstOrFail();

    $themeName = prepareFrontWebsiteData($website, $slug, true);

    return "{$themeName}::pages.page";
})->name('landing.page');

// Section Layout Preview Route - Used by admin panel for iframe previews
// URL: /section-preview/{sectionKey}/{layoutName}?website_id={xid}
Route::dynamicLivewire('/section-preview/{sectionKey}/{layoutName}', function (string $sectionKey, string $layoutName) {
    // Get website from query parameter if provided
    $websiteXid = request()->query('website_id');
    $themeName = 'tailwind_theme'; // Default theme

    if ($websiteXid && $websiteXid !== 'null') {
        $websiteId = \App\Classes\Common::getIdFromHash($websiteXid);
        $website = Website::withoutGlobalScopes()->find($websiteId);
        if ($website) {
            $themeName = $website->theme_name ?? 'tailwind_theme';
            prepareFrontWebsiteData($website, $website->slug);
        }
    }

    // Always share frontThemeName even if no website is selected
    view()->share('frontThemeName', $themeName);

    return "{$themeName}::pages.section-preview";
})->name('section.preview');
