<?php

use App\Livewire\LivewireBaseComponent;
use App\Models\WebsiteLegalPage;
use App\Models\WebsiteBuilder;

/**
 * Page Livewire Component
 *
 * Displays individual front pages (Privacy Policy, Terms of Service, Cookie Policy, etc.)
 * Extends LivewireBaseComponent to inherit common data (theme, seo, company, nav, social, footer, services)
 * Includes header and footer sections from website_builder
 */
return new class extends LivewireBaseComponent {
    // Page-specific properties
    public ?WebsiteLegalPage $page = null;

    // Header and Footer sections from website_builder
    public ?WebsiteBuilder $headerSection = null;
    public ?WebsiteBuilder $footerSection = null;

    // Flag to track if page was found
    public bool $pageNotFound = false;
    public string $notFoundReason = '';

    public function mount($slug = null, $pageSlug = '')
    {
        // Store website slug
        $this->websiteSlug = $slug;

        // Initialize common data (theme, seo, company, nav, social, footer, services)
        parent::mount($slug);

        // Check if website was found
        if (!$this->frontWebsite) {
            $this->pageNotFound = true;
            $this->notFoundReason = 'website';
        } else {
            // Load the page from database using the slug from route parameter
            $this->loadPage($pageSlug);
        }

        // Load header and footer sections
        $this->loadHeaderFooterSections();

        // Override SEO with page-specific data
        $this->overrideSeo();
    }

    /**
     * Load page from database by slug
     */
    protected function loadPage(string $slug): void
    {
        try {
            $query = WebsiteLegalPage::withoutGlobalScopes()->where('slug', $slug)->where('status', 'active');

            // Filter by website if we have one
            if ($this->frontWebsite) {
                $query->where('website_id', $this->frontWebsite->id);

                // Try to get page for current language first
                $page = (clone $query)->where('website_language_id', $this->currentLangId)->first();

                // Fallback to English if not found
                if (!$page && $this->currentLangKey !== 'en') {
                    $enLang = \App\Models\WebsiteLanguage::withoutGlobalScopes()->where('key', 'en')->where('company_id', $this->frontWebsite->company_id)->first();
                    $page = (clone $query)->where('website_language_id', $enLang?->id)->first();
                }

                // Final fallback: any language
                if (!$page) {
                    $page = $query->first();
                }

                if (!$page) {
                    $this->pageNotFound = true;
                    $this->notFoundReason = 'page';
                    return;
                }

                $this->page = $page;
            } else {
                // No website context - use company-level pages
                $page = $query->whereNull('website_id')->first();
                if (!$page) {
                    $this->pageNotFound = true;
                    $this->notFoundReason = 'page';
                    return;
                }
                $this->page = $page;
            }
        } catch (\Exception $e) {
            $this->pageNotFound = true;
            $this->notFoundReason = 'page';
        }
    }

    /**
     * Load header and footer sections from website_builder
     */
    protected function loadHeaderFooterSections(): void
    {
        try {
            $query = WebsiteBuilder::withoutGlobalScopes()->where('website_language_id', $this->currentLangId);

            if ($this->frontWebsite) {
                $query->where('website_id', $this->frontWebsite->id);
            }

            // Get header section
            $this->headerSection = (clone $query)->where('section_key', 'header')->first();

            // Fallback to English if no header found
            if (!$this->headerSection && $this->currentLangKey !== 'en' && $this->frontWebsite) {
                $enLang = \App\Models\WebsiteLanguage::withoutGlobalScopes()->where('key', 'en')->where('company_id', $this->frontWebsite->company_id)->first();
                $headerQuery = WebsiteBuilder::withoutGlobalScopes()->where('website_language_id', $enLang?->id)->where('section_key', 'header');
                if ($this->frontWebsite) {
                    $headerQuery->where('website_id', $this->frontWebsite->id);
                }
                $this->headerSection = $headerQuery->first();
            }

            // Get footer section
            $this->footerSection = (clone $query)->where('section_key', 'footer')->first();

            // Fallback to English if no footer found
            if (!$this->footerSection && $this->currentLangKey !== 'en' && $this->frontWebsite) {
                $enLang = \App\Models\WebsiteLanguage::withoutGlobalScopes()->where('key', 'en')->where('company_id', $this->frontWebsite->company_id)->first();
                $footerQuery = WebsiteBuilder::withoutGlobalScopes()->where('website_language_id', $enLang?->id)->where('section_key', 'footer');
                if ($this->frontWebsite) {
                    $footerQuery->where('website_id', $this->frontWebsite->id);
                }
                $this->footerSection = $footerQuery->first();
            }
        } catch (\Exception $e) {
            // Silently fail - header/footer will be null
        }
    }

    /**
     * Override SEO metadata with page-specific data
     */
    protected function overrideSeo(): void
    {
        if ($this->pageNotFound) {
            $this->seo = array_merge($this->seo, [
                'title' => 'Page Not Found',
                'description' => 'The requested page could not be found.',
                'keywords' => '',
            ]);
        } elseif ($this->page) {
            $this->seo = array_merge($this->seo, [
                'title' => $this->page->meta_title ?: $this->page->title,
                'description' => $this->page->meta_description ?: '',
                'keywords' => $this->page->meta_keywords ?: '',
            ]);
        }
    }

    /**
     * Get component name for a section
     */
    public function getComponentName(WebsiteBuilder $section): string
    {
        $sectionKey = $section->section_key;
        $layout = $section->section_layout ?? $sectionKey;

        return $this->themeName . '::components.' . $sectionKey . '.' . $layout;
    }

    /**
     * Get standardized props for any section component
     */
    public function getSectionProps(WebsiteBuilder $section): array
    {
        // Ensure content is an array (handle JSON string case)
        $content = $section->content_with_urls ?? ($section->content ?? []);
        if (is_string($content)) {
            $content = json_decode($content, true) ?? [];
        }
        if (!is_array($content)) {
            $content = [];
        }

        // Ensure items is always an array
        $items = $content['items'] ?? [];
        if (is_string($items)) {
            $items = json_decode($items, true) ?? [];
        }
        if (!is_array($items)) {
            $items = [];
        }

        // Build the standardized section data
        $sectionData = array_merge(
            [
                'section_key' => $section->section_key,
                'section_layout' => $section->section_layout ?? $section->section_key,
                'title' => $section->title,
                'subtitle' => $section->subtitle,
            ],
            $content,
        );

        // For header sections on page.blade.php, force light text color mode
        // because the breadcrumb section below has a dark gradient background
        if ($section->section_key === 'header') {
            $sectionData['text_color_mode'] = 'light';
        }

        return [
            'section' => $sectionData,
            'items' => $items,
            'company' => $this->company,
            'nav' => $this->nav,
            'navItems' => $this->navItems,
            'social' => $this->social,
            'footer' => $this->footer, // Footer defaults from LivewireBaseComponent
            'services' => $this->services,
            'languages' => $this->languages,
            'currentLangKey' => $this->currentLangKey,
            'websiteSlug' => $this->websiteSlug,
            'isWhiteBackground' => $section->is_white_background ?? false,
            'paddingTop' => $section->padding_top,
            'paddingBottom' => $section->padding_bottom,
        ];
    }

    /**
     * Get the home route for this context
     * Uses global helper landing_home_url() for consistent URL generation
     */
    public function getHomeRoute(): string
    {
        return landing_home_url($this->frontWebsite, $this->websiteSlug);
    }

    /**
     * Get inline padding style for a section
     * Returns empty string if no custom padding is set
     */
    public function getSectionPaddingStyle(WebsiteBuilder $section): string
    {
        $styles = [];
        if ($section->padding_top !== null) {
            $styles[] = "padding-top: {$section->padding_top}px";
        }
        if ($section->padding_bottom !== null) {
            $styles[] = "padding-bottom: {$section->padding_bottom}px";
        }
        return !empty($styles) ? implode('; ', $styles) : '';
    }

    /**
     * Check if section has custom padding
     */
    public function hasCustomPadding(WebsiteBuilder $section): bool
    {
        return $section->padding_top !== null || $section->padding_bottom !== null;
    }

    /**
     * Get the translated "Home" page name
     * Returns the translated name from website_pages + website_page_translations
     * Falls back to "Home" if not found
     */
    public function getHomePageName(): string
    {
        try {
            // Look for the home page (page_key = 'home')
            $homePage = \App\Models\WebsitePage::withoutGlobalScopes()->where('page_key', 'home');

            if ($this->frontWebsite) {
                $homePage->where('website_id', $this->frontWebsite->id);
            }

            $homePage = $homePage->first();

            if (!$homePage) {
                return 'Home';
            }

            // Try to get translation for current language
            $translation = \App\Models\WebsitePageTranslation::withoutGlobalScopes()->where('website_page_id', $homePage->id)->where('website_language_id', $this->currentLangId)->first();

            if ($translation && !empty($translation->name)) {
                return $translation->name;
            }

            // Fallback to English
            if ($this->currentLangKey !== 'en' && $this->frontWebsite) {
                $enLang = \App\Models\WebsiteLanguage::withoutGlobalScopes()->where('key', 'en')->where('company_id', $this->frontWebsite->company_id)->first();

                if ($enLang) {
                    $translation = \App\Models\WebsitePageTranslation::withoutGlobalScopes()->where('website_page_id', $homePage->id)->where('website_language_id', $enLang->id)->first();

                    if ($translation && !empty($translation->name)) {
                        return $translation->name;
                    }
                }
            }

            // Final fallback to default page name or "Home"
            return $homePage->name ?? 'Home';
        } catch (\Exception $e) {
            return 'Home';
        }
    }

    protected function view($data = [])
    {
        return app('view')->file('/Users/rajeshkumawat/sites/ant4-laravel12-vue3/storage/framework/views/livewire/views/b3e3e070.blade.php', $data);
    }

    public function styleModuleSrc()
    {
        return '/Users/rajeshkumawat/sites/ant4-laravel12-vue3/storage/framework/views/livewire/styles/b3e3e070.css';
    }
};
