<?php

use Livewire\Component;
use App\Models\WebsitePage;
use App\Models\WebsitePageTranslation;
use App\Models\WebsiteLanguage;

return new class extends Component {
    public array $section = [];
    public array $company = [];
    public array $nav = [];
    public array $navItems = [];
    public array $languages = [];
    public ?string $currentLangKey = 'en';
    public ?string $websiteSlug = null;
    public string $homeUrl = '';
    public array $headerPages = []; // Business pages with show_in_header = true

    public function mount(array $section = [], array $company = [], array $nav = [], array $navItems = [], array $languages = [], ?string $currentLangKey = 'en', ?string $websiteSlug = null)
    {
        $this->section = $section;
        $this->company = $company;
        $this->nav = $nav;
        $this->navItems = $navItems;
        $this->languages = $languages;
        $this->currentLangKey = $currentLangKey;
        $this->websiteSlug = $websiteSlug;

        // Generate home URL using global helper (handles is_primary for root URL)
        $frontWebsite = view()->shared('frontWebsite');
        $this->homeUrl = landing_home_url($frontWebsite, $this->websiteSlug);

        // Load business pages with show_in_header = true for current language
        $this->loadHeaderPages($frontWebsite);
    }

    /**
     * Load business pages that should be shown in header for the current language
     */
    protected function loadHeaderPages($frontWebsite): void
    {
        if (!$frontWebsite) {
            return;
        }

        // Get the current language ID
        $frontLangId = view()->shared('frontLangId');

        if (!$frontLangId) {
            // Try to get from current lang key
            $lang = WebsiteLanguage::withoutGlobalScopes()
                ->where('key', $this->currentLangKey)
                ->where('company_id', $frontWebsite->company_id)
                ->first();
            $frontLangId = $lang?->id;
        }

        if (!$frontLangId) {
            return;
        }

        // Get business pages with translations that have show_in_header = true
        $pages = WebsitePage::withoutGlobalScopes()
            ->where('website_id', $frontWebsite->id)
            ->whereHas('translations', function ($query) use ($frontLangId) {
                $query->where('website_language_id', $frontLangId)
                    ->where('show_in_header', true);
            })
            ->with(['translations' => function ($query) use ($frontLangId) {
                $query->where('website_language_id', $frontLangId);
            }])
            ->get();

        // Build header pages array
        $this->headerPages = $pages->map(function ($page) {
            $translation = $page->translations->first();
            return [
                'page_key' => $page->page_key,
                'name' => $translation?->name ?? $page->page_key,
                'url' => $this->getPageUrl($page->page_key),
            ];
        })->toArray();
    }

    /**
     * Get the URL for a business page
     */
    protected function getPageUrl(string $pageKey): string
    {
        // Home page goes to home URL
        if ($pageKey === 'home') {
            return $this->homeUrl;
        }

        // Other pages append page_key to home URL
        return rtrim($this->homeUrl, '/') . '/' . $pageKey;
    }

    /**
     * Change the frontend language
     * Stores in cookie with slug-based key (like theme mode), then triggers page reload
     * Cookie key format: lang_{slug} or lang_default
     */
    public function changeLanguage(string $langKey): void
    {
        // Use slug-based cookie key (same pattern as theme mode: theme_{slug})
        $langCookieKey = 'lang_' . ($this->websiteSlug ?? 'default');

        // Set cookie for persistence (1 year)
        cookie()->queue($langCookieKey, $langKey, 60 * 24 * 365);

        // Dispatch browser event to reload the page
        $this->dispatch('language-changed');
    }

    protected function view($data = [])
    {
        return app('view')->file('/Users/rajeshkumawat/sites/ant4-laravel12-vue3/storage/framework/views/livewire/views/d2e59589.blade.php', $data);
    }

    public function scriptModuleSrc()
    {
        return '/Users/rajeshkumawat/sites/ant4-laravel12-vue3/storage/framework/views/livewire/scripts/d2e59589.js';
    }

    public function styleModuleSrc()
    {
        return '/Users/rajeshkumawat/sites/ant4-laravel12-vue3/storage/framework/views/livewire/styles/d2e59589.css';
    }
};
