
export function run($wire, $js) {
    (function() {
        function updateHeaderDarkMode() {
            const header = document.getElementById('header');
            if (!header) return;

            const isSolid = header.dataset.solid === 'true';
            const isTransparent = header.dataset.transparent === 'true';
            const isScrolled = header.classList.contains('header-scrolled');
            const isDark = document.body.classList.contains('dark');

            // Solid header (non-transparent)
            if (isSolid) {
                if (isDark) {
                    header.style.setProperty('background', '#0f172a', 'important');
                    header.style.setProperty('border-bottom', '1px solid rgba(255, 255, 255, 0.1)', 'important');
                } else {
                    header.style.setProperty('background', 'rgba(255, 255, 255, 0.95)', 'important');
                    header.style.setProperty('border-bottom', 'none', 'important');
                }
            }
            // Transparent header when scrolled
            else if (isTransparent && isScrolled) {
                if (isDark) {
                    header.style.setProperty('background', '#0f172a', 'important');
                    header.style.setProperty('border-bottom', '1px solid rgba(255, 255, 255, 0.1)', 'important');
                } else {
                    header.style.setProperty('background', 'rgba(255, 255, 255, 0.95)', 'important');
                    header.style.setProperty('border-bottom', 'none', 'important');
                }
            }
            // Transparent header not scrolled - keep transparent
            else if (isTransparent && !isScrolled) {
                header.style.removeProperty('background');
                header.style.removeProperty('border-bottom');
            }
        }

        // Run IMMEDIATELY since header element exists at this point (script is after header)
        updateHeaderDarkMode();

        // Also run on DOMContentLoaded as a fallback
        document.addEventListener('DOMContentLoaded', updateHeaderDarkMode);

        // Watch for dark mode changes on body
        const bodyObserver = new MutationObserver(function(mutations) {
            mutations.forEach(function(mutation) {
                if (mutation.attributeName === 'class') {
                    updateHeaderDarkMode();
                }
            });
        });

        // Watch for scroll state changes on header (header-scrolled class)
        function observeHeader() {
            const header = document.getElementById('header');
            if (header) {
                const headerObserver = new MutationObserver(function(mutations) {
                    mutations.forEach(function(mutation) {
                        if (mutation.attributeName === 'class') {
                            updateHeaderDarkMode();
                        }
                    });
                });
                headerObserver.observe(header, { attributes: true });
            }
        }

        // Start observing body for class changes
        if (document.body) {
            bodyObserver.observe(document.body, { attributes: true });
            observeHeader();
        } else {
            document.addEventListener('DOMContentLoaded', function() {
                bodyObserver.observe(document.body, { attributes: true });
                observeHeader();
            });
        }

        // Also run on Livewire navigation
        document.addEventListener('livewire:navigated', function() {
            updateHeaderDarkMode();
            observeHeader();
        });

        // Run on scroll to catch scroll state changes
        window.addEventListener('scroll', updateHeaderDarkMode);
    })();
}