import { defineConfig, loadEnv } from 'vite';
import laravel from 'laravel-vite-plugin';
import tailwindcss from '@tailwindcss/vite';
import path from 'path';
import { fileURLToPath } from 'url';
import { globSync } from 'glob';

const __dirname = path.dirname(fileURLToPath(import.meta.url));

export default defineConfig(({ mode }) => {
    // Load env file based on `mode` in the current working directory.
    const env = loadEnv(mode, process.cwd(), '');

    // Get theme name from .env file, default to 'tailwind_theme'
    const themeName = env.VITE_FRONT_THEME || 'tailwind_theme';

    // Dynamically get all CSS files from components folder
    const componentCssFiles = globSync(`resources/views/themes/${themeName}/assets/components/**/*.css`);

    return {
        cacheDir: `node_modules/.vite-${themeName}`,
        build: {
            outDir: `public/build-${themeName}`,
            emptyOutDir: true,
            manifest: 'manifest.json',
        },
        resolve: {
            alias: {
                tailwindcss: path.resolve(__dirname, 'node_modules/@tailwindcss/vite/node_modules/tailwindcss'),
            },
        },
        plugins: [
            tailwindcss(),
            laravel({
                publicDirectory: 'public',
                buildDirectory: `build-${themeName}`,
                hotFile: `storage/vite-${themeName}.hot`,
                input: [
                    `resources/views/themes/${themeName}/assets/app.css`,
                    `resources/views/themes/${themeName}/assets/app.js`,
                    // Dynamically include all component CSS files
                    ...componentCssFiles,
                ],
                refresh: true,
            }),
        ],
    };
});
