import path from "path";
import { defineConfig, loadEnv } from "vite";
import laravel from "laravel-vite-plugin";
import vue from "@vitejs/plugin-vue";
import Components from "unplugin-vue-components/vite";
import { AntDesignVueResolver } from "unplugin-vue-components/resolvers";
import tailwindcss from "@tailwindcss/vite";
import fs from "fs";

export default defineConfig(({ mode }) => {
    // Load .env file based on current mode (e.g., .env, .env.development, etc.)
    const env = loadEnv(mode, process.cwd());

    // Generate aliases for all modules
    const modulesPath = path.resolve(__dirname, "./Modules");
    const moduleAliases = {};

    if (fs.existsSync(modulesPath)) {
        const modules = fs.readdirSync(modulesPath, { withFileTypes: true })
            .filter(dirent => dirent.isDirectory())
            .map(dirent => dirent.name);

        modules.forEach(moduleName => {
            const moduleJsPath = path.resolve(modulesPath, moduleName, "resources/assets/js");
            if (fs.existsSync(moduleJsPath)) {
                const aliasName = `@${moduleName.toLowerCase()}`;
                moduleAliases[aliasName] = moduleJsPath;
            }
        });

        // Log module aliases in development mode
        if (mode === 'development' && Object.keys(moduleAliases).length > 0) {
            console.log('📦 Module aliases registered:', Object.keys(moduleAliases).join(', '));
        }
    }

    return {
        css: {
            preprocessorOptions: {
                less: {
                    javascriptEnabled: true, // Enable inline JavaScript in Less
                },
            },
        },
        plugins: [
            tailwindcss(),
            laravel({
                input: [
                    "resources/less/tailwind.css",
                    "resources/js/app.js",
                    "resources/less/custom.less"
                ],
            }),
            vue({
                template: {
                    transformAssetUrls: {
                        base: null,
                        includeAbsolute: false,
                    },
                },
            }),
            Components({
                resolvers: [
                    AntDesignVueResolver({
                        importStyle: false, // css in js
                    }),
                ],
            }),
        ],
        resolve: {
            alias: {
                "@": path.resolve(__dirname, "./resources/js"),
                tailwindcss: path.resolve(__dirname, "./node_modules/@tailwindcss/vite/node_modules/tailwindcss"),
                ...moduleAliases, // Dynamically add all module aliases
            },
        },
        build: {
            // Enable minification
            minify: 'terser',
            terserOptions: {
                compress: {
                    drop_console: false, // Remove console.log in production
                    drop_debugger: true,
                },
            },
            // Optimize chunk size
            rollupOptions: {
                output: {
                    manualChunks(id) {
                        // Split node_modules by package for better caching
                        if (id.includes('node_modules')) {
                            // Core Vue packages
                            if (id.includes('node_modules/vue/') || id.includes('@vue/')) {
                                return 'vue-core';
                            }
                            // Vue Router - separate chunk
                            if (id.includes('vue-router')) {
                                return 'vue-router';
                            }
                            // Pinia - separate chunk
                            if (id.includes('pinia')) {
                                return 'pinia';
                            }
                            // Axios - small utility
                            if (id.includes('axios')) {
                                return 'axios';
                            }
                            // Ant Design Vue - large library, separate chunk
                            if (id.includes('ant-design-vue')) {
                                return 'antd-vue';
                            }
                            // Ant Design Icons - separate chunk (can be large)
                            if (id.includes('@ant-design/icons-vue')) {
                                return 'antd-icons';
                            }
                            // i18n related
                            if (id.includes('vue-i18n') || id.includes('@intlify')) {
                                return 'i18n';
                            }
                            // Chart/graph libraries
                            if (id.includes('chart') || id.includes('echarts')) {
                                return 'charts';
                            }
                            // Other large vendors
                            const match = id.match(/node_modules\/([^/]+)/);
                            if (match) {
                                const packageName = match[1];
                                // Group small packages together
                                if (['lodash', 'moment', 'dayjs'].includes(packageName)) {
                                    return 'utils';
                                }
                            }
                            // All other vendors
                            return 'vendor';
                        }

                        // Layouts and components
                        if (id.includes('layouts/AdminLayout')) {
                            return 'layout-admin';
                        }
                        if (id.includes('components/Sidebar')) {
                            return 'sidebar';
                        }

                        // Settings pages (lazy loaded)
                        if (id.includes('views/settings') || id.includes('views/common/settings')) {
                            return 'settings-pages';
                        }

                        // User management pages
                        if (id.includes('views/users')) {
                            return 'users-pages';
                        }
                    },
                },
            },
            chunkSizeWarningLimit: 600,
            // Source maps only in dev
            sourcemap: mode === 'development',
        },
        server: {
            hmr: {
                host: env.VITE_HRM_HOST ?? 'localhost',
            },
        },
    }
});
